import { toast } from 'react-toastify';
import { GetServerSideProps, NextPage } from 'next';
import { useTranslation } from 'next-i18next';
import { serverSideTranslations } from 'next-i18next/serverSideTranslations';
import { useRouter } from 'next/router';
import { useMutation } from '@tanstack/react-query';
import { Button, Form } from 'antd';
import { RoleForm } from '@/components/role/RoleForm';
import Layout from '@/components/layouts';
import { createRole } from '@/services/role';
import { BasePageProps, Role } from '@/types';
import { authentication } from '@/utils/authentication';
import { errorMessageFormatter } from '@/utils/errorFormatter';
import localeLinkGenerator from '@/utils/localeLinkGenerator';

const AddRolePage: NextPage<BasePageProps> = ({ staff }) => {
    const { t } = useTranslation(['role', 'common', 'staff']);
    const [addRoleForm] = Form.useForm();
    const router = useRouter();

    const createRoleMutation = useMutation({
        mutationFn: async (role: Role) => {
            const res = await createRole(role);
            return res.data;
        },
        onSuccess: (data) => {
            router.push('/role/' + data.id);
        },
    });

    const onCreateRoleHandler = () => {
        addRoleForm.validateFields().then(() => {
            toast.promise(createRoleMutation.mutateAsync(addRoleForm.getFieldsValue()), {
                pending: {
                    render() {
                        return t('messages:creating-role');
                    },
                },
                success: {
                    render() {
                        return t('messages:role-created');
                    },
                },
                error: {
                    render({ data }: any) {
                        return t(errorMessageFormatter(data));
                    },
                },
            });
        });
    };

    const breadCrumbItems = [
        {
            label: t('role'),
            path: '/role',
        },
        {
            label: t('add'),
            path: '/role/add',
        },
    ];

    const seoConfig = {
        title: t('add-role'),
    };

    return (
        <Layout staff={staff} activeMenu={['role']} breadCrumbItems={breadCrumbItems} seoConfig={seoConfig} withBackground>
            <div>
                <RoleForm form={addRoleForm} />
                <div className="flex justify-end gap-2">
                    <Button loading={createRoleMutation.isLoading} type="primary" onClick={onCreateRoleHandler}>
                        {t('add-role')}
                    </Button>
                </div>
            </div>
        </Layout>
    );
};

export default AddRolePage;

export const getServerSideProps: GetServerSideProps = async ({ locale, req, resolvedUrl }) => {
    try {
        const authResponse = await authentication(req, 'ROLE_CREATE');

        return {
            props: {
                staff: authResponse,
                ...(await serverSideTranslations(locale as string, ['role', 'layout', 'common', 'messages', 'staff', 'api-messages'])),
            },
        };
    } catch (error: any) {
        if (error.response?.data?.unauthorized) {
            return {
                redirect: {
                    destination: localeLinkGenerator(locale, `/unauthorized`),
                    permanent: false,
                },
            };
        }

        return {
            redirect: {
                destination: localeLinkGenerator(locale, `/?redirect=/${resolvedUrl}`),
                permanent: false,
            },
        };
    }
};
