import { STANDARD_DATE_FORMAT_WITHOUT_TIME } from '@/configs';
import { StudyGroup, StudyGroupStatus } from '@/types';
import { Editor } from '@tinymce/tinymce-react';
import { Col, DatePicker, Form, Input, Row } from 'antd';
import dayjs from 'dayjs';
import { useTranslation } from 'next-i18next';
import { v4 as uuid } from 'uuid';

interface EditGroupActionProps {
    studyGroupId: string | undefined;
    studyGroupData: StudyGroup | undefined;
}

const EditGroupForm: React.FC<EditGroupActionProps> = ({ studyGroupData }) => {
    const { t, i18n } = useTranslation(['study-group', 'layout', 'common', 'messages']);
    const currentLocale = i18n.language;

    const disabledDate = (current: any) => {
        return current && current < dayjs.tz().endOf('day');
    };

    return (
        <Row gutter={[16, 16]} className="mb-2">
            <Col span={24}>
                <Form.Item name="name" label={t('group-name')} rules={[{ required: true, message: t('messages:required') }]}>
                    <Input placeholder={t('enter-group-name')} size="large" />
                </Form.Item>
                <Form.Item name="groupPeriod" label={t('groupPeriod')} rules={[{ required: true, message: t('messages:required') }]}>
                    <DatePicker.RangePicker
                        format={STANDARD_DATE_FORMAT_WITHOUT_TIME}
                        size="large"
                        disabled={studyGroupData?.status !== StudyGroupStatus.DRAFT}
                        className="w-full"
                        disabledDate={disabledDate}
                    />
                </Form.Item>
                <Form.Item name="description" label={t('description')} trigger="onEditorChange" className="mb-2">
                    <Editor
                        id={uuid()}
                        tinymceScriptSrc={'/tinymce/js/tinymce/tinymce.min.js'}
                        init={{
                            height: 300,
                            menubar: '',
                            plugins: ['advlist', 'autolink', 'lists', 'link', 'searchreplace', 'code', 'preview', 'wordcount', 'table'],
                            toolbar:
                                'undo redo | blocks | ' +
                                'bold italic forecolor | link | alignleft aligncenter ' +
                                'alignright alignjustify | bullist numlist outdent indent | ' +
                                'fullscreen',
                            block_formats: 'Paragraph=p;Header 2=h2;Header 3=h3;Header 4=h4',
                            content_style: 'body { font-family:var(--font-ubuntu); font-size:14px; text-align:left }',
                            language: currentLocale === 'en-GB' ? undefined : currentLocale,
                            language_url: currentLocale === 'en-GB' ? undefined : '/editor-translation/' + currentLocale + '.js',
                            promotion: false,
                            paste_data_images: true,
                            contextmenu: false,
                        }}
                    />
                </Form.Item>
            </Col>
        </Row>
    );
};

export default EditGroupForm;
