import ConfirmationModal from '@/components/modals/ConfirmationModal';
import { STANDARD_DATE_FORMAT_WITHOUT_TIME } from '@/configs';
import { useStaffContext } from '@/providers/StaffContext';
import { useStudyGroupContext } from '@/providers/StudyGroupContext';
import { AxiosErrorResponse, CommentLike, MemberData, StudyGroupMember, StudyGroupTaskComment } from '@/types';
import { errorMessageFormatter, stringAvatar, stringToColor } from '@/utils';
import { CommentOutlined, EllipsisOutlined, LikeFilled, LikeOutlined, MedicineBoxOutlined, WhatsAppOutlined } from '@ant-design/icons';
import { useMutation, useQuery, useQueryClient } from '@tanstack/react-query';
import { Avatar, Button, Card, Divider, Dropdown, Empty, Form, MenuProps, Pagination, Spin, Tooltip } from 'antd';
import dayjs from 'dayjs';
import Link from 'next/link';
import { useEffect, useState } from 'react';
import { Trans, useTranslation } from 'next-i18next';
import { toast } from 'react-toastify';
import usePagination from '@/hooks/usePagination';
import { extractStudyGroupMembers } from '@/transformer/study-group';
import {
    getStudyGroupTaskComments,
    getCommentsByTaskId,
    deleteCommentById,
    likeComment,
    getOwnTaskComment,
    getTaskCommentLikes,
} from '@/services/study-group';
import HtmlParser from '@/components/HtmlParser';
import { dateTimeTransformer } from '@/utils/timeTransformer';
import FilterDrawer from '../modals/CommentFilter';
import ReplyModal from '../modals/Reply';
import RenderReply from '../RenderReply';

const CommentTab: React.FC = () => {
    const { t } = useTranslation(['study-group-task', 'common']);
    const [commentId, setCommentId] = useState<string>('');
    const { staff } = useStaffContext();
    const { studyGroupTaskId, studyGroupMember, studyGroupTask } = useStudyGroupContext();
    const [mobileWidth, setMobileWidth] = useState<boolean>();
    const [pagination, setPagination] = usePagination();
    const [filterCommentForm] = Form.useForm();
    const queryClient = useQueryClient();

    // State to track expanded comments and replies
    const [expandedCommentId, setExpandedCommentId] = useState<string | null>();
    const [expandedCommentIdArray, setExpandedCommentIdArray] = useState<string[]>([]);

    // State to track actions (Comment / Reply)
    const [action, setAction] = useState<string>();

    // Query
    const commentsQuery = useQuery({
        queryKey: ['comments', studyGroupTaskId, 'pagination', pagination],
        enabled: !!studyGroupTaskId,

        queryFn: async () => {
            const filterValue = filterCommentForm.getFieldsValue();

            const query = {
                ...filterValue,
                page: pagination.page,
                pageSize: pagination.pageSize,
                sortField: pagination.sortField,
                sortOrder: pagination.sortOrder,
            };

            const res = await getStudyGroupTaskComments(studyGroupTaskId, query);

            const pageTotal = res.data?.total - res.data?.rows.filter((comment: StudyGroupTaskComment) => comment.parent_id).length;

            setPagination((prev) => {
                return {
                    ...prev,
                    page: res.data?.page,
                    total: pageTotal,
                };
            });

            return res.data?.rows;
        },
        onError: (error: AxiosErrorResponse & Error) => {
            toast.error(t(errorMessageFormatter(error)));
        },
    });

    const commentsData = commentsQuery.data;

    const { data: commentsByTaskId, refetch: commentsByTaskIdRefetch } = useQuery({
        queryKey: ['task-comments', studyGroupTaskId],
        enabled: !!studyGroupTaskId,
        queryFn: async () => {
            const res = await getCommentsByTaskId(studyGroupTaskId);

            return res.data;
        },
    });

    const studyGroupMembers = extractStudyGroupMembers(studyGroupMember.data);

    const memberWhoHaventCommented = studyGroupMembers?.filter((member: StudyGroupMember) => {
        return (
            !Array.isArray(commentsByTaskId) ||
            !commentsByTaskId.find((comment: StudyGroupTaskComment) => {
                return comment?.memberId === member?.id;
            })
        );
    });

    const ownCommentQuery = useQuery({
        queryKey: ['own-comment', studyGroupTaskId],
        enabled: !!studyGroupTaskId,
        queryFn: async () => {
            const res = await getOwnTaskComment(studyGroupTaskId);
            return res.data;
        },
        onError: (error: AxiosErrorResponse & Error) => {
            toast.error(t(errorMessageFormatter(error)));
        },
    });
    const ownCommentData = ownCommentQuery.data;

    // own member comment is always first while the other follow and remove ones with parent_id
    const commentList = (
        ownCommentData && commentsData ? (ownCommentData.prePlanned ? commentsData : [ownCommentData, ...commentsData]) : commentsData
    )?.filter((comment: StudyGroupTaskComment) => !comment.parent_id);

    const commentLikesQuery = useQuery({
        queryKey: ['comment-likes', studyGroupTaskId],
        enabled: !!studyGroupTaskId,
        queryFn: async () => {
            const res = await getTaskCommentLikes(studyGroupTaskId);

            return res.data;
        },
        onError: (error: AxiosErrorResponse & Error) => {
            toast.error(t(errorMessageFormatter(error)));
        },
    });
    const commentLikesData = commentLikesQuery.data;

    // Mutation
    const deleteCommentMutation = useMutation({
        mutationFn: async () => {
            const res = await deleteCommentById(commentId);
            return res.data;
        },
        onSuccess: async () => {
            commentsByTaskIdRefetch();

            commentsQuery.refetch();
        },
    });

    const likeCommentMutation = useMutation({
        mutationFn: async (studyGroupTaskCommentId: string) => {
            const response = await likeComment(studyGroupTaskCommentId, staff?.id as string, studyGroupTaskId);

            return response.data;
        },
        onSuccess: () => {
            queryClient.invalidateQueries(['own-comment']);
            ownCommentQuery.refetch();
            commentsQuery.refetch();
            studyGroupTask.refetch();
            commentLikesQuery.refetch();
        },
    });

    // Function
    useEffect(() => {
        return () => {
            if (window.innerWidth < 800) {
                setMobileWidth(true);
            } else {
                setMobileWidth(false);
            }
        };
    });

    const gridStyleBody: React.CSSProperties = {
        width: '100%',
        textAlign: 'start',
        paddingBottom: 0,
    };

    const gridStyleFooter: React.CSSProperties = {
        width: '100%',
        textAlign: 'end',
        paddingTop: 16,
        paddingBottom: 16,
    };

    const onDeleteHandler = () => {
        toast.promise(deleteCommentMutation.mutateAsync(), {
            pending: t('messages:deleting-comment'),
            success: t('messages:comment-deleted'),
            error: {
                render({ data }) {
                    return t(errorMessageFormatter(data as AxiosErrorResponse));
                },
            },
        });
    };

    const handleLike = async (studyGroupTaskCommentId: string) => {
        // Check if the comment is already liked by the user
        const alreadyLiked = commentLikesData?.some(
            (like: CommentLike) => like.studyGroupTaskCommentId === studyGroupTaskCommentId && like.staffId === staff?.id,
        );

        // Perform like/unlike action based on the current state
        if (alreadyLiked) {
            // Unlike the comment
            await toast.promise(likeCommentMutation.mutateAsync(studyGroupTaskCommentId), {
                pending: t('common:unliking-comment'),
                success: t('common:comment-unliked'),
                error: {
                    render({ data }) {
                        return t(errorMessageFormatter(data as AxiosErrorResponse));
                    },
                },
            });
        } else {
            // Like the comment
            await toast.promise(likeCommentMutation.mutateAsync(studyGroupTaskCommentId), {
                pending: t('common:liking-comment'),
                success: t('common:comment-liked'),
                error: {
                    render({ data }) {
                        return t(errorMessageFormatter(data as AxiosErrorResponse));
                    },
                },
            });
        }
    };

    // Update the like count and icon based on the current state of likes
    const renderLikeButton = (comment: StudyGroupTaskComment) => {
        const liked = commentLikesData?.some((like: CommentLike) => like.studyGroupTaskCommentId === comment.id && like.staffId === staff?.id);
        const likesCount = commentLikesData?.filter((like: CommentLike) => like.studyGroupTaskCommentId === comment.id).length;

        return (
            <div className="flex flex-row gap-1 items-center">
                <Tooltip title={liked ? t('common:unlike') : t('common:like')}>
                    <Button
                        type="text"
                        shape="circle"
                        size="middle"
                        icon={liked ? <LikeFilled style={{ color: '#597ef7' }} /> : <LikeOutlined />}
                        onClick={() => handleLike(comment.id)}
                        loading={likeCommentMutation.isLoading}
                    />
                </Tooltip>
                <span className="text-xs sm:text-sm ml-1">{likesCount}</span>
            </div>
        );
    };

    const handleSetExpandedCommentId = (commentId: string) => {
        setExpandedCommentId(commentId === expandedCommentId ? null : commentId);
        setExpandedCommentIdArray(
            expandedCommentIdArray.includes(commentId)
                ? expandedCommentIdArray.filter((id) => id !== commentId)
                : [...expandedCommentIdArray, commentId],
        );
    };

    const onResetHandler = () => {
        setAction('');
    };

    // Data
    const items: MenuProps['items'] = [
        {
            key: 'delete',
            danger: true,
            label: (
                <ConfirmationModal
                    title={t('common:delete-confirmation')}
                    message={<Trans i18nKey={'messages:are-you-sure-you-want-to-delete'} components={{ strong: <strong /> }} />}
                    okText={t('common:delete')}
                    onOk={onDeleteHandler}
                    reason={false}
                    okButtonProps={{
                        danger: true,
                    }}
                    width={400}
                >
                    <Button
                        loading={deleteCommentMutation.isLoading}
                        type="link"
                        ghost
                        block
                        className="!p-0 !w-20 hover:!text-white text-black"
                        style={{ transition: 'none' }}
                    >
                        {t('common:delete')}
                    </Button>
                </ConfirmationModal>
            ),
        },
    ];

    return (
        <div>
            {!mobileWidth ? (
                <div className="p-4 pt-0">
                    <div className="grid grid-cols-3 gap-4">
                        <Card title={t('comments')} className="col-span-2">
                            <Card.Grid style={gridStyleBody} hoverable={false}>
                                {commentList && commentList.length !== 0 ? (
                                    commentList.map((comment: StudyGroupTaskComment) => (
                                        <div
                                            key={comment.id}
                                            className="flex flex-col gap-2 mb-6 border-solid rounded-xl border-[#f5f5f5] hover:shadow-md"
                                        >
                                            <div className="p-4 pb-0">
                                                <div className="flex flex-row gap-4">
                                                    <div className="flex flex-col w-full mb-2">
                                                        <div className="flex flex-row justify-between gap-2">
                                                            <div className="flex flex-row gap-2 mb-2">
                                                                {comment.member?.preferredName ? (
                                                                    <div className="font-semibold flex flex-row items-center gap-2">
                                                                        <Avatar
                                                                            style={{ backgroundColor: stringToColor(comment.member?.preferredName) }}
                                                                            size={30}
                                                                            className="flex items-center flex-row gap-2"
                                                                        >
                                                                            {stringAvatar(comment.member?.preferredName)}
                                                                        </Avatar>
                                                                        <span className="text-xs sm:text-sm">{comment.member?.preferredName}</span>
                                                                    </div>
                                                                ) : (
                                                                    <div className="font-semibold flex flex-row items-center gap-2">
                                                                        <Avatar
                                                                            style={{ backgroundColor: stringToColor(comment.member?.fullName) }}
                                                                            size={30}
                                                                            className="flex items-center flex-row gap-2"
                                                                        >
                                                                            {stringAvatar(comment.member?.fullName)}
                                                                        </Avatar>
                                                                        <span className="text-xs sm:text-sm">{comment.member?.fullName}</span>
                                                                    </div>
                                                                )}
                                                                <div className="text-gray-400 flex items-center text-xs">
                                                                    {dateTimeTransformer(comment.commentDate)}
                                                                </div>
                                                            </div>
                                                            {staff?.role.STUDY_GROUP_TASK_DELETE && (
                                                                <Dropdown trigger={['click']} menu={{ items }}>
                                                                    <EllipsisOutlined
                                                                        style={{ cursor: 'pointer', rotate: '90deg', marginTop: '7px' }}
                                                                        onClick={() => setCommentId(comment.id)}
                                                                    />
                                                                </Dropdown>
                                                            )}
                                                        </div>
                                                        <div className="text-xs sm:text-sm ml-[38px]">
                                                            <HtmlParser html={comment.comment} />
                                                        </div>
                                                    </div>
                                                </div>
                                                <Divider className="my-0" />
                                                <div className="flex flex-row items-center gap-x-8">
                                                    <div className="flex flex-row gap-1 align-middle items-center py-1">
                                                        {renderLikeButton(comment)}
                                                    </div>
                                                    <div className="flex flex-row gap-2">
                                                        <Button
                                                            type="text"
                                                            icon={<CommentOutlined />}
                                                            onClick={() => handleSetExpandedCommentId(comment.id)}
                                                        >
                                                            {t('replies') + ` (${comment.Children.length})`}
                                                        </Button>
                                                    </div>
                                                </div>
                                            </div>
                                            {expandedCommentIdArray.find((commentId: string) => {
                                                return commentId === comment.id;
                                            }) && (
                                                <div className="p-4 pb-0 bg-gray-50 rounded-lg rounded-t-none">
                                                    <div className="flex flex-col sm:flex-row sm:justify-between gap-4 sm:gap-0 mb-4">
                                                        <div className="text-[#343434] text-lg flex items-center">
                                                            <b>{t('replies') + ` (${comment.Children.length})`}</b>
                                                        </div>
                                                        <ReplyModal
                                                            onReset={onResetHandler}
                                                            studyGroupTaskId={studyGroupTaskId}
                                                            parentId={comment.id}
                                                        />
                                                    </div>
                                                    <RenderReply studyGroupTaskId={studyGroupTaskId} commentId={comment.id} />
                                                </div>
                                            )}
                                        </div>
                                    ))
                                ) : (
                                    <Empty description={t('common:no-comments-available')} className="mb-4" />
                                )}
                            </Card.Grid>
                            {commentList && commentList.length !== 0 ? (
                                <Card.Grid style={gridStyleFooter} hoverable={false}>
                                    <div className="flex justify-end">
                                        <Pagination
                                            current={pagination.page}
                                            pageSize={pagination.pageSize}
                                            defaultPageSize={1}
                                            showSizeChanger={true}
                                            pageSizeOptions={[10, 25, 50, 100]}
                                            // total={commentList.length}
                                            total={pagination.total}
                                            showTotal={(total, range) => {
                                                return t('common:pagination', { range0: range[0], range1: range[1], total: total });
                                            }}
                                            onChange={(page, pageSize) => setPagination({ ...pagination, page, pageSize })}
                                        />
                                    </div>
                                </Card.Grid>
                            ) : null}
                        </Card>
                        <Card title={t('members-without-comments')}>
                            {memberWhoHaventCommented?.length > 0 ? (
                                memberWhoHaventCommented.map((member: MemberData) => {
                                    return (
                                        <div key={member?.id} className="flex flex-col gap-2 mb-2">
                                            <div className="flex flex-row justify-between">
                                                <div className="flex flex-col gap-2 w-full">
                                                    <div className="flex flex-row justify-between gap-2">
                                                        <div className="flex flex-row gap-2 mb-2 w-full">
                                                            <div className="w-full font-semibold flex flex-row items-center gap-2 justify-between">
                                                                <div className="flex flex-row items-center ">
                                                                    {member?.preferredName ? (
                                                                        <div className="flex flex-row items-center">
                                                                            <Avatar
                                                                                style={{ backgroundColor: stringToColor(member?.preferredName) }}
                                                                                size={30}
                                                                                className="flex items-center flex-row gap-2"
                                                                            >
                                                                                {stringAvatar(member?.preferredName)}
                                                                            </Avatar>
                                                                            <span className="text-xs sm:text-sm ml-2">{member?.preferredName}</span>
                                                                        </div>
                                                                    ) : (
                                                                        <div className="flex flex-row items-center">
                                                                            <Avatar
                                                                                style={{ backgroundColor: stringToColor(member?.fullName) }}
                                                                                size={30}
                                                                                className="flex items-center flex-row gap-2"
                                                                            >
                                                                                {stringAvatar(member?.fullName)}
                                                                            </Avatar>
                                                                            <span className="text-xs sm:text-sm ml-2">{member?.fullName}</span>
                                                                        </div>
                                                                    )}
                                                                    {member.medicalCertificate?.length > 0 &&
                                                                        member.medicalCertificate[0].status === 'APPROVED' &&
                                                                        dayjs
                                                                            .tz(member.medicalCertificate[0].applyDate)
                                                                            .format(STANDARD_DATE_FORMAT_WITHOUT_TIME) ===
                                                                            dayjs
                                                                                .tz(studyGroupTask.data?.taskDate)
                                                                                .format(STANDARD_DATE_FORMAT_WITHOUT_TIME) && (
                                                                            <MedicineBoxOutlined className="text-blue-400 ml-4 pb-[2px]" />
                                                                        )}
                                                                </div>
                                                                <div>
                                                                    <Link
                                                                        target="_blank"
                                                                        href={`https://wa.me/${member?.phoneNumber}`}
                                                                        onClick={(e) => e.stopPropagation()}
                                                                        className="flex items-center"
                                                                    >
                                                                        <WhatsAppOutlined className="text-lg text-green-500 hover:text-green-400 " />
                                                                    </Link>
                                                                </div>
                                                            </div>
                                                        </div>
                                                    </div>
                                                </div>
                                            </div>
                                        </div>
                                    );
                                })
                            ) : memberWhoHaventCommented?.length === 0 && commentsData?.length ? (
                                <Empty description={t('all-members-commented')} />
                            ) : (
                                <Empty description={t('no-available-members')} />
                            )}
                        </Card>
                    </div>
                </div>
            ) : (
                <div className="flex flex-col gap-4">
                    <div className="flex flex-col">
                        <Card.Grid style={gridStyleBody} hoverable={false}>
                            <div className="text-lg font-bold mb-2 ml-1">{t('comments')}</div>
                            {commentList && commentList.length !== 0 ? (
                                commentList.map((comment: StudyGroupTaskComment) => {
                                    return <div
                                        key={comment.id}
                                        className="flex flex-col gap-2 mb-6 border-solid rounded-xl border-[#f5f5f5] hover:shadow-md"
                                    >
                                        <div className="p-4 pb-0">
                                            <div className="flex flex-row gap-4">
                                                <div className="flex flex-col w-full mb-2">
                                                    <div className="flex flex-row justify-between gap-2">
                                                        <div className="flex flex-row gap-2 mb-2">
                                                            {comment.member?.preferredName || comment.staff?.fullName ? (
                                                                <div className="font-semibold flex flex-row items-center gap-2">
                                                                    <Avatar
                                                                        style={{ backgroundColor: stringToColor(comment.member?.preferredName) }}
                                                                        size={30}
                                                                        className="flex items-center flex-row gap-2"
                                                                    >
                                                                        {stringAvatar(comment.member?.preferredName)}
                                                                    </Avatar>
                                                                    <span className="text-xs sm:text-sm">{comment.member?.preferredName}</span>
                                                                </div>
                                                            ) : (
                                                                <div className="font-semibold flex flex-row items-center gap-2">
                                                                    <Avatar
                                                                        style={{ backgroundColor: stringToColor(comment.member?.fullName) }}
                                                                        size={30}
                                                                        className="flex items-center flex-row gap-2"
                                                                    >
                                                                        {stringAvatar(comment.member?.fullName)}
                                                                    </Avatar>
                                                                    <span className="text-xs sm:text-sm">{comment.member?.fullName}</span>
                                                                </div>
                                                            )}
                                                            <div className="text-gray-400 flex items-center text-xs">
                                                                {dateTimeTransformer(comment.commentDate)}
                                                            </div>
                                                        </div>
                                                        {staff?.role.STUDY_GROUP_TASK_DELETE && (
                                                            <Dropdown trigger={['click']} menu={{ items }}>
                                                                <EllipsisOutlined
                                                                    style={{ cursor: 'pointer', rotate: '90deg', marginTop: '7px' }}
                                                                    onClick={() => setCommentId(comment.id)}
                                                                />
                                                            </Dropdown>
                                                        )}
                                                    </div>
                                                    <div className="text-xs sm:text-sm ml-[38px]">
                                                        <HtmlParser html={comment.comment} />
                                                    </div>
                                                </div>
                                            </div>
                                            <Divider className="my-0" />
                                            <div className="flex flex-row items-center gap-x-8">
                                                <div className="flex flex-row gap-1 align-middle items-center py-1">{renderLikeButton(comment)}</div>
                                                <div className="flex flex-row gap-2">
                                                    <Button
                                                        type="text"
                                                        icon={<CommentOutlined />}
                                                        onClick={() => handleSetExpandedCommentId(comment.id)}
                                                    >
                                                        {t('replies') + ` (${comment.Children.length})`}
                                                    </Button>
                                                </div>
                                            </div>
                                        </div>
                                        {expandedCommentIdArray.find((commentId: string) => {
                                            return commentId === comment.id;
                                        }) && (
                                            <div className="p-4 pb-0 bg-gray-50 rounded-lg rounded-t-none">
                                                <div className="flex flex-col sm:flex-row sm:justify-between gap-4 sm:gap-0 mb-4">
                                                    <div className="text-[#343434] text-lg flex items-center">
                                                        <b>{t('replies') + ` (${comment.Children.length})`}</b>
                                                    </div>
                                                    <ReplyModal onReset={onResetHandler} studyGroupTaskId={studyGroupTaskId} parentId={comment.id} />
                                                </div>
                                                <RenderReply studyGroupTaskId={studyGroupTaskId} commentId={comment.id} />
                                            </div>
                                        )}
                                    </div>
                                })
                            ) : (
                                <Empty description={t('common:no-comments-available')} className="mb-4" />
                            )}
                        </Card.Grid>
                        {commentList && commentList.length !== 0 ? (
                            <Card.Grid style={gridStyleFooter} hoverable={false} className="!pt-0">
                                <div className="flex justify-end">
                                    <Pagination
                                        current={pagination.page}
                                        pageSize={pagination.pageSize}
                                        defaultPageSize={1}
                                        showSizeChanger={true}
                                        pageSizeOptions={[10, 25, 50, 100]}
                                        total={pagination.total}
                                        showTotal={(total, range) => {
                                            return t('common:pagination', { range0: range[0], range1: range[1], total: total });
                                        }}
                                        onChange={(page, pageSize) => setPagination({ ...pagination, page, pageSize })}
                                    />
                                </div>
                            </Card.Grid>
                        ) : null}
                    </div>

                    <Card title={t('members-without-comments')}>
                        {memberWhoHaventCommented?.length > 0 ? (
                            memberWhoHaventCommented.map((member: MemberData) => {
                                return (
                                    <div key={member?.id} className="flex flex-col gap-2 mb-2">
                                        <div className="flex flex-row justify-between">
                                            <div className="flex flex-col gap-2 w-full">
                                                <div className="flex flex-row justify-between gap-2">
                                                    <div className="flex flex-row gap-2 mb-2 w-full">
                                                        <div className="w-full font-semibold flex flex-row items-center gap-2 justify-between">
                                                            <div className="flex flex-row items-center ">
                                                                {member?.preferredName ? (
                                                                    <div className="flex flex-row items-center ">
                                                                        <Avatar
                                                                            style={{ backgroundColor: stringToColor(member?.preferredName) }}
                                                                            size={30}
                                                                            className="flex items-center flex-row gap-2"
                                                                        >
                                                                            {stringAvatar(member?.preferredName)}
                                                                        </Avatar>
                                                                        <span className="text-xs sm:text-sm ml-2">{member?.preferredName}</span>
                                                                    </div>
                                                                ) : (
                                                                    <div>
                                                                        <Avatar
                                                                            style={{ backgroundColor: stringToColor(member?.fullName) }}
                                                                            size={30}
                                                                            className="flex items-center flex-row gap-2"
                                                                        >
                                                                            {stringAvatar(member?.fullName)}
                                                                        </Avatar>
                                                                        <span className="text-xs sm:text-sm ml-2">{member?.fullName}</span>
                                                                    </div>
                                                                )}
                                                                {member.medicalCertificate?.length > 0 &&
                                                                    member.medicalCertificate[0].status === 'APPROVED' &&
                                                                    dayjs
                                                                        .tz(member.medicalCertificate[0].applyDate)
                                                                        .format(STANDARD_DATE_FORMAT_WITHOUT_TIME) ===
                                                                        dayjs
                                                                            .tz(studyGroupTask.data?.taskDate)
                                                                            .format(STANDARD_DATE_FORMAT_WITHOUT_TIME) && (
                                                                        <MedicineBoxOutlined className="text-blue-400 ml-4 pb-[2px]" />
                                                                    )}
                                                            </div>
                                                            <div>
                                                                <Link
                                                                    target="_blank"
                                                                    href={`https://wa.me/${member?.phoneNumber}`}
                                                                    onClick={(e) => e.stopPropagation()}
                                                                    className="flex items-center"
                                                                >
                                                                    <WhatsAppOutlined className="text-lg text-green-500 hover:text-green-400 " />
                                                                </Link>
                                                            </div>
                                                        </div>
                                                    </div>
                                                </div>
                                            </div>
                                        </div>
                                    </div>
                                );
                            })
                        ) : memberWhoHaventCommented?.length === 0 && commentsData?.length ? (
                            <Empty description={t('all-members-commented')} />
                        ) : (
                            <Empty description={t('no-available-members')} />
                        )}
                    </Card>
                </div>
            )}
        </div>
    );
};

export default CommentTab;
