import { useEffect } from 'react';
import { toast } from 'react-toastify';
import { useTranslation } from 'next-i18next';
import { UserOutlined } from '@ant-design/icons';
import { UseQueryResult, useMutation } from '@tanstack/react-query';
import { Button, Form, Input, Spin } from 'antd';
import { useStaffContext } from '@/providers/StaffContext';
import { updateStaffProfile } from '@/services/staff';
import { Staff, UpdateStaffProfileParams } from '@/types';
import { errorMessageFormatter } from '@/utils/errorFormatter';

interface ProfileProps {
    staffId: string;
    staffQuery: UseQueryResult<Staff | null>;
}

const Profile: React.FC<ProfileProps> = ({ staffId, staffQuery }) => {
    const { t } = useTranslation(['staff', 'common']);
    const { staff } = useStaffContext();
    const [staffForm] = Form.useForm();
    const staffToBeEdit = staffQuery.data;

    useEffect(() => {
        if (staffToBeEdit) {
            staffForm.setFieldsValue({
                ...staffToBeEdit,
            });
        }
    }, [staffToBeEdit]);

    // Mutation
    const updateStaffProfileMutation = useMutation({
        mutationFn: async ({ staffId, body }: { staffId: string; body: UpdateStaffProfileParams }) => {
            const res = await updateStaffProfile(staffId, body);
            return res.data;
        },
        onSuccess: () => {
            staffQuery.refetch();
        },
    });

    // Function
    const onUpdateStaffProfileHandler = () => {
        staffForm.validateFields().then(async (values) => {
            toast.promise(updateStaffProfileMutation.mutateAsync({ staffId, body: values }), {
                pending: t('messages:updating-staff-profile'),
                success: t('messages:staff-profile-updated'),
                error: {
                    render({ data }: any) {
                        return t(errorMessageFormatter(data));
                    },
                },
            });
        });
    };

    return (
        <Spin spinning={updateStaffProfileMutation.isLoading}>
            {staff && (
                <div className="flex flex-col lg:flex-row">
                    <div className="order-2 w-full px-2 mt-10 lg:mt-0 lg:flex-1 lg:order-1">
                        <Form form={staffForm} layout="vertical" title="Staff Form">
                            {/* Full Name Input */}
                            <Form.Item
                                label={t('full-name')}
                                name="fullName"
                                rules={[
                                    {
                                        required: true,
                                    },
                                ]}
                                className="mb-5"
                                labelCol={{ flex: '35px' }}
                            >
                                <Input classNames={{ prefix: '!me-2' }} prefix={<UserOutlined className="site-form-item-icon" />} />
                            </Form.Item>
                            {staff?.role.STAFF_UPDATE && (
                                <div className="flex justify-end">
                                    <Button
                                        loading={updateStaffProfileMutation.isLoading}
                                        type="primary"
                                        className="hover:!text-white"
                                        style={{ transition: 'none' }}
                                        onClick={onUpdateStaffProfileHandler}
                                    >
                                        {t('common:update')}
                                    </Button>
                                </div>
                            )}
                        </Form>
                    </div>
                </div>
            )}
        </Spin>
    );
};

export default Profile;
