import React, { useState } from 'react';
import { toast } from 'react-toastify';
import { useTranslation } from 'next-i18next';
import { FilterOutlined, SearchOutlined } from '@ant-design/icons';
import { useQuery } from '@tanstack/react-query';
import { Button, Col, Form, Input, Modal, Row, Select } from 'antd';
import { FormInstance } from 'antd/es/form/Form';
import { getRoleData } from '@/services/data';
import { AxiosErrorResponse } from '@/types';
import { errorMessageFormatter } from '@/utils/errorFormatter';

interface FilterAttributes {
    filterStaffForm: FormInstance;
    onReset: () => void;
    onSearch: () => void;
    loading: boolean;
}

const FilterDrawer: React.FC<FilterAttributes> = ({ filterStaffForm, onReset, onSearch, loading }) => {
    const { t } = useTranslation(['staff', 'common']);
    const criteriaSelected = filterStaffForm.getFieldsValue();
    const criteriaCount: number = Object.values(criteriaSelected).reduce((count: number, val) => count + (val ? 1 : 0), 0);
    const [isModalOpen, setIsModalOpen] = useState(false);

    const breakPoint = {
        xs: 24,
        sm: 24,
        md: 12,
        lg: 12,
    };

    const roleQuery = useQuery({
        queryKey: ['role', 'data'],
        queryFn: async () => {
            const res = await getRoleData();
            return res.data;
        },
        onError: (error: AxiosErrorResponse & Error) => {
            toast.error(errorMessageFormatter(error));
        },
        staleTime: 1000 * 60 * 5, // 5 minutes
    });

    const roleSelection =
        roleQuery.data?.map((role) => ({
            label: role.name,
            value: role.id,
        })) ?? [];

    const onResetHandler = () => {
        onReset();
        setIsModalOpen(false);
    };

    const onSearchHandler = () => {
        onSearch();
        setIsModalOpen(false);
    };

    return (
        <div>
            <Button onClick={() => setIsModalOpen(true)} icon={<FilterOutlined />}>
                {criteriaCount > 0 && `(${criteriaCount})`} {t('common:filter')}
            </Button>
            <Modal title={t('common:filter')} onCancel={onResetHandler} open={isModalOpen} footer={null} width={650}>
                <Form form={filterStaffForm} layout="vertical" name="filter_form" className="mt-6" size="large">
                    <Row gutter={[16, 0]}>
                        <Col {...breakPoint}>
                            <Form.Item initialValue="" name="fullNameEmail" label={t('full-name-email')} labelCol={{ flex: '35px' }}>
                                <Input placeholder={t('full-name-or-email')} />
                            </Form.Item>
                        </Col>
                        <Col {...breakPoint}>
                            <Form.Item name="role" label={t('role')} labelCol={{ flex: '35px' }} className="mb-5">
                                <Select placeholder={t('common:select-role')} options={roleSelection} />
                            </Form.Item>
                        </Col>
                    </Row>
                    <div className="flex justify-between mt-3 gap-x-3">
                        <Button onClick={onResetHandler} block loading={loading}>
                            {t('common:reset')}
                        </Button>
                        <Button type="primary" onClick={onSearchHandler} block loading={loading}>
                            {t('common:apply-filter')}
                            <SearchOutlined />
                        </Button>
                    </div>
                </Form>
            </Modal>
        </div>
    );
};

export default FilterDrawer;
