import { useStaffContext } from '@/providers/StaffContext';
import { getPenaltyAmount, updatePenaltyAmount } from '@/services/site-setting';
import { AxiosErrorResponse, PenaltyAmount } from '@/types';
import { currencyFormatter, currencyParser, errorMessageFormatter } from '@/utils';
import { useMutation, useQuery } from '@tanstack/react-query';
import { Button, Form, InputNumber } from 'antd';
import { useTranslation } from 'next-i18next';
import { toast } from 'react-toastify';

const PenaltySettingTab: React.FC = () => {
    const { t } = useTranslation(['site-setting', 'layout', 'common', 'messages']);
    const [penaltyForm] = Form.useForm();
    const { staff } = useStaffContext();

    // Query
    useQuery({
        queryKey: ['penalty-amount'],
        queryFn: async () => {
            const response = await getPenaltyAmount();
            return response.data;
        },
        onSuccess(data) {
            penaltyForm.setFieldsValue({
                penaltyAmount: data.penaltyAmount,
            });
        },
    });

    const savePenaltyAmountMutation = useMutation({
        mutationFn: async (body: PenaltyAmount) => {
            const response = await updatePenaltyAmount(body);
            return response.data;
        },
    });

    const savePenaltyAmountHandler = async () => {
        penaltyForm.validateFields().then(async (values) => {
            toast.promise(savePenaltyAmountMutation.mutateAsync(values), {
                pending: t('messages:updating-penalty-amount'),
                success: t('messages:penalty-amount-updated'),
                error: {
                    render({ data }) {
                        return t(errorMessageFormatter(data as AxiosErrorResponse));
                    },
                },
            });
        });
    };

    return (
        <Form form={penaltyForm} layout="vertical">
            <Form.Item label={t('penalty-amount')} name="penaltyAmount" rules={[{ required: true, message: t('messages:required') }]}>
                <InputNumber min={0} prefix={'MYR'} style={{ width: '100%' }} formatter={currencyFormatter} parser={currencyParser} />
            </Form.Item>
            {staff?.role.SITE_SETTING && (
                <div className="flex justify-end">
                    <Button loading={savePenaltyAmountMutation.isLoading} onClick={savePenaltyAmountHandler} type="primary">
                        {t('common:save')}
                    </Button>
                </div>
            )}
        </Form>
    );
};

export default PenaltySettingTab;
