import { useEffect, useState } from 'react';
import { MdOutlineEdit } from 'react-icons/md';
import { useTranslation } from 'next-i18next';
import { UseQueryResult } from '@tanstack/react-query';
import { Button, Input, Spin, Typography } from 'antd';
import { useStaffContext } from '@/providers/StaffContext';
import { Member } from '@/types';
import EmailModal from '../modals/EmailModal';
import PasswordModal from '../modals/PasswordModal';
import PhoneNumberModal from '../modals/PhoneModal';

interface CredentialTabProps {
    memberQuery: UseQueryResult<Member | null>;
}

const Credential: React.FC<CredentialTabProps> = ({ memberQuery }) => {
    const { t } = useTranslation(['member', 'common']);
    const { staff } = useStaffContext();
    const memberToBeEdit = memberQuery.data;
    const { Title } = Typography;

    const [email, setEmail] = useState<string>('');
    const [phoneNumber, setPhoneNumber] = useState<string>('');
    const [isEmailModalOpen, setIsEmailModalOpen] = useState<boolean>(false);
    const [isPhoneNumberModalOpen, setIsPhoneNumberModalOpen] = useState<boolean>(false);
    const [isPasswordModalOpen, setIsPasswordModalOpen] = useState<boolean>(false);

    useEffect(() => {
        if (memberToBeEdit) {
            setEmail(memberToBeEdit.email);
            setPhoneNumber(memberToBeEdit.phoneNumber);
        }
    }, [memberToBeEdit]);

    return (
        <Spin spinning={memberQuery.isLoading}>
            {memberToBeEdit && (
                <div>
                    {/* Email Input */}
                    <div className="mb-6">
                        <Title level={5}>{t('email')}</Title>
                        <div className="flex gap-2 items-center">
                            <Input name="email" disabled value={email} className="max-w-[300px]" />
                            {staff?.role.MEMBER_UPDATE && (
                                <Button
                                    type="primary"
                                    onClick={() => {
                                        setIsEmailModalOpen(true);
                                    }}
                                >
                                    <MdOutlineEdit className="text-xl" />
                                </Button>
                            )}
                        </div>
                    </div>
                    {/* Phone Number Input */}
                    <div className="mb-6">
                        <Title level={5}>{t('phone-number')}</Title>
                        <div className="flex gap-2 items-center">
                            <Input name="phoneNumber" disabled value={phoneNumber} className="max-w-[300px]" />
                            {staff?.role.MEMBER_UPDATE && (
                                <Button
                                    type="primary"
                                    onClick={() => {
                                        setIsPhoneNumberModalOpen(true);
                                    }}
                                >
                                    <MdOutlineEdit className="text-xl" />
                                </Button>
                            )}
                        </div>
                    </div>
                    {/* Password Input */}
                    <div className="mb-4">
                        <Title level={5}>{t('password')}</Title>
                        <div className="flex gap-2 items-center">
                            <Input.Password disabled defaultValue="************" className="max-w-[300px]" visibilityToggle={false} />
                            <Button type="primary" onClick={() => setIsPasswordModalOpen(true)}>
                                <MdOutlineEdit className="text-xl" />
                            </Button>
                        </div>
                    </div>
                    <div>
                        <EmailModal open={isEmailModalOpen} setOpen={setIsEmailModalOpen} member={memberToBeEdit} memberQuery={memberQuery} />
                        <PhoneNumberModal
                            open={isPhoneNumberModalOpen}
                            setOpen={setIsPhoneNumberModalOpen}
                            member={memberToBeEdit}
                            memberQuery={memberQuery}
                        />
                        <PasswordModal open={isPasswordModalOpen} setOpen={setIsPasswordModalOpen} member={memberToBeEdit} memberQuery={memberQuery} />
                    </div>
                </div>
            )}
        </Spin>
    );
};

export default Credential;
