import { useEffect } from 'react';
import PhoneInput from 'react-phone-input-2';
import 'react-phone-input-2/lib/style.css';
import { toast } from 'react-toastify';
import { useTranslation } from 'next-i18next';
import { UseQueryResult, useMutation } from '@tanstack/react-query';
import { Button, Form, Modal } from 'antd';
import { updateMemberPhoneNumber } from '@/services/member';
import { Member, BaseModalProps, UpdateStaffPhoneNumberParams, AxiosErrorResponse } from '@/types';
import { errorMessageFormatter } from '@/utils/errorFormatter';

interface PhoneNumberModalProps extends BaseModalProps {
    member: Member;
    memberQuery: UseQueryResult<Member | null>;
}

const PhoneNumberModal: React.FC<PhoneNumberModalProps> = ({ open, setOpen, member, memberQuery }) => {
    const { t } = useTranslation(['member', 'common']);
    const [form] = Form.useForm<{ phoneNumber: string }>();
    const phoneNumberInput = Form.useWatch('phoneNumber', form);

    useEffect(() => {
        form.setFieldsValue({ phoneNumber: member.phoneNumber });
    }, []);

    const isDiasbled = () => {
        if (!phoneNumberInput || phoneNumberInput === member.phoneNumber) {
            return true;
        }

        return false;
    };

    // Mutations
    const updateMemberPhoneNumberMutation = useMutation({
        mutationFn: async ({ memberId, body }: { memberId: string; body: UpdateStaffPhoneNumberParams }) => {
            const res = await updateMemberPhoneNumber(memberId, body);
            return res.data;
        },
        onSuccess: () => {
            setOpen(false);
            memberQuery.refetch();
        },
    });

    // Functions
    const onCancelHandler = () => {
        form.resetFields();
        setOpen(false);
    };

    const onUpdateMemberPhoneNumberHandler = () => {
        form.validateFields().then(async (values) => {
            toast.promise(updateMemberPhoneNumberMutation.mutateAsync({ memberId: member.id, body: values }), {
                pending: t('messages:updating-member-phone-number'),
                success: t('messages:member-phone-number-updated'),
                error: {
                    render({ data }) {
                        return t(errorMessageFormatter(data as AxiosErrorResponse));
                    },
                },
            });
        });
    };

    return (
        <Modal open={open} onCancel={onCancelHandler} footer={null} title={t('edit-phone-number')}>
            <Form layout="vertical" form={form} className="mt-4">
                <Form.Item label={t('phone-number')} name="phoneNumber" rules={[{ required: true }]} labelCol={{ flex: '35px' }}>
                    <PhoneInput
                        value={phoneNumberInput}
                        inputProps={{
                            required: true,
                        }}
                        inputClass="form-control"
                        country={'my'}
                        inputStyle={{ width: '100%', height: '40px', borderRadius: '8px' }}
                        buttonStyle={{ borderRadius: '8px 0px 0px 8px' }}
                    />
                </Form.Item>
                <div className="pt-2 flex gap-3">
                    <Button loading={updateMemberPhoneNumberMutation.isLoading} size="large" block onClick={onCancelHandler}>
                        {t('common:cancel')}
                    </Button>
                    <Button
                        loading={updateMemberPhoneNumberMutation.isLoading}
                        size="large"
                        type="primary"
                        disabled={isDiasbled()}
                        block
                        onClick={onUpdateMemberPhoneNumberHandler}
                    >
                        {t('common:update')}
                    </Button>
                </div>
            </Form>
        </Modal>
    );
};

export default PhoneNumberModal;
