import PhoneInput from 'react-phone-input-2';
import 'react-phone-input-2/lib/style.css';
import { toast } from 'react-toastify';
import { useTranslation } from 'next-i18next';
import { EnvironmentOutlined, MailOutlined, UserOutlined } from '@ant-design/icons';
import { UseQueryResult, useMutation, useQueryClient } from '@tanstack/react-query';
import { Button, Col, DatePicker, Form, Input, Modal, Row } from 'antd';
import { createMember } from '@/services/member';
import { Member, BaseModalProps, CreateMemberParams, AxiosErrorResponse } from '@/types';
import { errorMessageFormatter } from '@/utils/errorFormatter';
import dayjs from 'dayjs';

interface AddMemberModalProps extends BaseModalProps {
    memberQuery: UseQueryResult<Member[] | null>;
}

const AddMember: React.FC<AddMemberModalProps> = ({ open, setOpen, memberQuery }) => {
    const { t } = useTranslation(['member', 'common']);
    const queryClient = useQueryClient();
    const [addMemberForm] = Form.useForm<CreateMemberParams>();

    // Mutation
    const createMemberMutation = useMutation({
        mutationFn: async (params: CreateMemberParams) => {
            const res = await createMember(params);
            return res.data;
        },
        onSuccess: () => {
            setOpen(false); // Close Add Member Modal
            addMemberForm.resetFields(); // Reset Add Member Form
            memberQuery.refetch(); // Refetch Member Query
            queryClient.invalidateQueries(['member'], { exact: true }); // Disable Member Caching
        },
    });

    // Functions
    const onCreateMemberHandler = async () => {
        addMemberForm.validateFields().then(async (values) => {
            toast.promise(createMemberMutation.mutateAsync(values), {
                pending: t('messages:creating-member'),
                success: t('messages:member-created'),
                error: {
                    render({ data }) {
                        return t(errorMessageFormatter(data as AxiosErrorResponse));
                    },
                },
            });
        });
    };

    const onCancelHandler = () => {
        addMemberForm.resetFields();
        setOpen(false);
    };

    const breakPoint = {
        xs: 24,
        sm: 24,
        md: 12,
        lg: 12,
    };

    const disabledDate = (current: any) => {
        return current && current > dayjs.tz().endOf('day');
    };

    return (
        <Modal title={t('add-member')} open={open} onCancel={onCancelHandler} footer={null} width={650}>
            <Form form={addMemberForm} layout="vertical" size="large" name="add_member_form" className="mt-4">
                {/* Full Name Input */}
                <Form.Item
                    label={t('full-name')}
                    name="fullName"
                    rules={[
                        {
                            required: true,
                        },
                    ]}
                    className="mb-5"
                    labelCol={{ flex: '35px' }}
                >
                    <Input classNames={{ prefix: '!me-2' }} prefix={<UserOutlined className="site-form-item-icon" />} />
                </Form.Item>
                {/* Preferred Name Input */}
                <Form.Item label={t('preferred-name')} name="preferredName" labelCol={{ flex: '35px' }}>
                    <Input classNames={{ prefix: '!me-2' }} prefix={<UserOutlined className="site-form-item-icon" />} />
                </Form.Item>
                {/* Address Input */}
                <Form.Item
                    label={t('address')}
                    name="address"
                    rules={[
                        {
                            required: true,
                        },
                    ]}
                    labelCol={{ flex: '35px' }}
                >
                    <Input classNames={{ prefix: '!me-2' }} prefix={<EnvironmentOutlined className="site-form-item-icon" />} />
                </Form.Item>
                <Row gutter={[16, 0]}>
                    {/* Email Input */}
                    <Col {...breakPoint}>
                        <Form.Item
                            label={t('email')}
                            name="email"
                            rules={[
                                {
                                    required: true,
                                },
                                { type: 'email' },
                            ]}
                            labelCol={{ flex: '35px' }}
                        >
                            <Input classNames={{ prefix: '!me-2' }} prefix={<MailOutlined className="site-form-item-icon" />} />
                        </Form.Item>
                    </Col>
                    {/* Date of Birth Input */}
                    <Col {...breakPoint}>
                        <Form.Item
                            label={t('date-of-birth')}
                            name="dateOfBirth"
                            rules={[
                                {
                                    required: true,
                                },
                            ]}
                            labelCol={{ flex: '35px' }}
                        >
                            <DatePicker size="large" className="w-full" format="DD MMM YYYY" disabledDate={disabledDate} />
                        </Form.Item>
                    </Col>
                </Row>
                <Row gutter={[16, 0]}>
                    {/* Phone Number Input */}
                    <Col {...breakPoint}>
                        <Form.Item label={t('phone-number')} name="phoneNumber" rules={[{ required: true }]} labelCol={{ flex: '35px' }}>
                            <PhoneInput
                                inputProps={{
                                    required: true,
                                }}
                                inputClass="form-control"
                                country={'my'}
                                inputStyle={{ width: '100%', height: '40px', borderRadius: '8px' }}
                                buttonStyle={{ borderRadius: '8px 0px 0px 8px' }}
                            />
                        </Form.Item>
                    </Col>
                </Row>
                <div className="flex justify-between mt-3 gap-x-3">
                    <Button type="default" block onClick={onCancelHandler}>
                        {t('common:cancel')}
                    </Button>
                    <Button type="primary" disabled={createMemberMutation.isLoading} block htmlType="submit" onClick={onCreateMemberHandler}>
                        {t('common:add')}
                    </Button>
                </div>
            </Form>
        </Modal>
    );
};

export default AddMember;
