import React, { useEffect, useState } from 'react';
import { DefaultSeoProps, NextSeo } from 'next-seo';
import Image from 'next/image';
import Link from 'next/link';
import { Layout as AntLayout, Breadcrumb, Drawer } from 'antd';
import { useStaffContext } from '@/providers/StaffContext';
import { BreadCrumbItem, Staff } from '@/types';
import Logo from '../../../public/logo-text.png';
import { Header } from './Header';
import { Menu } from './Menu';
import { useTranslation } from 'next-i18next';

const { Sider, Content } = AntLayout;

interface LayoutAttributes {
    children: React.ReactNode;
    staff: Staff;
    breadCrumbItems?: BreadCrumbItem[];
    activeMenu?: string[];
    activeDropdown?: string[];
    seoConfig?: DefaultSeoProps;
    withBackground?: boolean;
}

const Layout: React.FC<LayoutAttributes> = ({
    children,
    staff,
    breadCrumbItems = [],
    activeMenu = [],
    activeDropdown = [],
    seoConfig,
    withBackground = false,
}) => {
    const { t } = useTranslation(['layout']);

    const [collapsed, setCollapsed] = useState(false);
    const [isMobile, setIsMobile] = useState<boolean>(false);
    const [isDrawerOpen, setIsDrawerOpen] = useState<boolean>(false);

    const { setStaff } = useStaffContext();

    useEffect(() => {
        setStaff(staff);
    }, []);

    // Check if the user is on mobile
    useEffect(() => {
        const isMobile = window.innerWidth <= 768;
        setIsMobile(isMobile);
        window.addEventListener('resize', () => {
            const isMobile = window.innerWidth <= 768;
            if (isMobile) {
                setCollapsed(true);
            }
            setIsMobile(isMobile);
        });
    }, []);

    const breadcrumbItems = [
        {
            title: <Link href="/dashboard">{t('home')}</Link>,
        },
    ];

    for (let i = 0; i < breadCrumbItems.length; i++) {
        const item = breadCrumbItems[i];
        if (i === breadCrumbItems.length - 1) {
            breadcrumbItems.push({
                title: <span className="font-semibold">{item.label}</span>,
            });
        } else {
            breadcrumbItems.push({
                title: <Link href={item.path}>{item.label}</Link>,
            });
        }
    }

    return (
        <div className="min-h-screen">
            <NextSeo {...seoConfig} />
            <AntLayout className="min-h-screen">
                <Header
                    isMobile={isMobile}
                    isDrawerOpen={isDrawerOpen}
                    setIsDrawerOpen={setIsDrawerOpen}
                    collapsed={collapsed}
                    setCollapsed={setCollapsed}
                    fullName={staff.fullName}
                />
                <div className="mt-[64px]"></div>
                <AntLayout hasSider style={{ minHeight: '100%' }}>
                    {isMobile ? (
                        <Drawer
                            placement="left"
                            open={isDrawerOpen}
                            onClose={() => setIsDrawerOpen(false)}
                            width="250px"
                            styles={{
                                body: {
                                    padding: 0,
                                },
                            }}
                            closable={false}
                        >
                            <div className="flex flex-col">
                                <div className="px-10 py-4">
                                    <Image src={Logo} alt="Image Logo" className="w-full h-full" />
                                </div>
                                <Menu activeMenu={activeMenu} activeDropdown={activeDropdown} />
                            </div>
                        </Drawer>
                    ) : (
                        <>
                            <div className={collapsed ? 'duration-200 ml-[80px]' : 'duration-200 ml-[250px]'}></div>
                            <Sider
                                trigger={null}
                                collapsible
                                collapsed={collapsed}
                                theme="light"
                                width={250}
                                style={{
                                    overflow: 'scroll',
                                    position: 'fixed',
                                    marginTop: '64px',
                                    left: 0,
                                    top: 0,
                                    bottom: 0,
                                    zIndex: 100,
                                }}
                            >
                                <Menu activeMenu={activeMenu} activeDropdown={activeDropdown} />
                            </Sider>
                        </>
                    )}
                    <Content className="p-9">
                        <div>
                            {/* Breadcrumb */}
                            <Breadcrumb className="!mb-3" separator=">" items={breadcrumbItems} />
                            <div className={`${withBackground && 'bg-white p-6'}`}>{children}</div>
                        </div>
                    </Content>
                </AntLayout>
            </AntLayout>
        </div>
    );
};

export default Layout;
