import { useMemo, useRef, useState } from 'react';
import { MdOutlineKeyboardAlt } from 'react-icons/md';
import { useTranslation } from 'next-i18next';
import { Empty, Select, Spin } from 'antd';
import type { SelectProps } from 'antd/es/select';
import debounce from 'lodash/debounce';

export interface DebounceSelectProps<ValueType = any> extends Omit<SelectProps<ValueType | ValueType[]>, 'options' | 'children' | 'showSearch'> {
    fetchOptions: (search: string) => Promise<ValueType[]>;
    debounceTimeout?: number;
}

const DebounceSelect: React.FC<DebounceSelectProps> = <T extends { key?: string; label: React.ReactNode; value: string | number } = any>({
    fetchOptions,
    debounceTimeout = 500,
    ...props
}: DebounceSelectProps<T>) => {
    const { t } = useTranslation(['common']);
    const [fetching, setFetching] = useState(false);
    const [options, setOptions] = useState<T[]>([]);
    const [searchValue, setSearchValue] = useState<string>('');
    const fetchRef = useRef(0);

    const debounceFetcher = useMemo(() => {
        const loadOptions = (value: string) => {
            if (!value) {
                setFetching(false);
                return;
            }
            fetchRef.current += 1;
            const fetchId = fetchRef.current;

            fetchOptions(value).then((newOptions) => {
                if (fetchId !== fetchRef.current) {
                    // for fetch callback order
                    return;
                }

                setOptions(newOptions);
                setFetching(false);
            });
        };
        return debounce(loadOptions, debounceTimeout);
    }, [fetchOptions, debounceTimeout]);

    const notFoundContent = fetching ? (
        <div className="py-2 flex justify-center">
            <Spin size="small" />
        </div>
    ) : !searchValue ? (
        <div className="py-2 flex flex-col justify-center items-center">
            <MdOutlineKeyboardAlt className="text-xl" />
            <p className="text-black text-opacity-40">{t('start-typing-to-search')}</p>
        </div>
    ) : (
        <Empty />
    );

    return (
        <Select
            className="!w-full"
            labelInValue
            size={props.size}
            filterOption={false}
            onSearch={(value) => {
                setOptions([]);
                setSearchValue(value);
                if (value === undefined) return;
                setFetching(true);
                debounceFetcher(value);
            }}
            notFoundContent={notFoundContent}
            {...props}
            options={options}
            showSearch
            loading={fetching}
            searchValue={searchValue}
        />
    );
};

export default DebounceSelect;
