import usePagination from '@/hooks/usePagination';
import { AxiosErrorResponse, BasePageProps, StudyGroup } from '@/types';
import { authentication, errorMessageFormatter, localeLinkGenerator } from '@/utils';
import { Button, Form, Table, TableColumnProps, Tag } from 'antd';
import { useTranslation } from 'next-i18next';
import { toast } from 'react-toastify';
import { ColumnsType } from 'antd/es/table';
import Layout from '@/components/layouts';
import { GetServerSideProps } from 'next';
import { serverSideTranslations } from 'next-i18next/serverSideTranslations';
import { useQuery } from '@tanstack/react-query';
import FilterDrawer from '@/components/study-group/modals/Filter';
import Link from 'next/link';
import { getStudyGroupList } from '@/services/study-group';
import ExportData from '@/components/study-group/ExportData';
import { dateTimeTransformer, dateTransformer } from '@/utils/timeTransformer';

const StudyGroupPage: React.FC<BasePageProps> = ({ member }) => {
    const { t } = useTranslation(['study-group', 'common', 'messages', 'layout']);
    const [pagination, setPagination, paginationOnChange] = usePagination();
    const [filterStudyGroupForm] = Form.useForm();

    // Query
    const studyGroupListQuery = useQuery({
        queryKey: ['study-group', 'pagination', pagination],
        queryFn: async () => {
            const filterValues = filterStudyGroupForm.getFieldsValue();

            const query = {
                ...filterValues,
                page: pagination.page,
                pageSize: pagination.pageSize,
                sortField: pagination.sortField,
                sortOrder: pagination.sortOrder,
            };

            const response = await getStudyGroupList(query);

            setPagination((prev) => {
                return {
                    ...prev,
                    page: response.data?.page,
                    total: response.data?.total,
                };
            });

            return response.data?.rows;
        },
        onError: (error: AxiosErrorResponse & Error) => {
            toast.error(t(errorMessageFormatter(error)));
        },
    });

    // Functions
    const onResetFilterHandler = () => {
        filterStudyGroupForm.resetFields();
        studyGroupListQuery.refetch();
    };

    // Data
    const columns: ColumnsType<StudyGroup> = [
        {
            title: t('group-name'),
            dataIndex: 'name',
            key: 'name',
            render: (_: unknown, studyGroup) => {
                return (
                    <Link href={`/study-group/${studyGroup.id}`}>
                        <b>{studyGroup.name}</b>
                    </Link>
                );
            },
        },
        {
            title: t('book-name'),
            dataIndex: 'book',
            key: 'book',
            render: (_: unknown, studyGroup) => {
                return studyGroup.book?.name;
            },
        },
        {
            title: t('members'),
            dataIndex: 'members',
            key: 'members',
            width: 120,
            render: (_: unknown, studyGroup: StudyGroup) => {
                return studyGroup._count?.studyGroupMembers;
            },
        },
        {
            title: t('start-date'),
            dataIndex: 'startDate',
            key: 'startDate',
            render: (_: unknown, studyGroup) => {
                return dateTransformer(studyGroup.startDate);
            },
        },
        {
            title: t('end-date'),
            dataIndex: 'endDate',
            key: 'endDate',
            render: (_: unknown, studyGroup) => {
                return dateTransformer(studyGroup.endDate);
            },
        },
        {
            title: t('status'),
            dataIndex: 'status',
            key: 'status',
            width: 130,
            render: (_: unknown, studyGroup) => {
                let tagColor, tagText;

                switch (studyGroup?.status) {
                    case 'DRAFT':
                        tagColor = '#FFA500';
                        tagText = t('preparing');
                        break;
                    case 'GENERATED':
                        tagColor = '#FFA500';
                        tagText = t('preparing');
                        break;
                    case 'ONGOING':
                        tagColor = '#FFD700';
                        tagText = t('ongoing');
                        break;
                    case 'COMPLETED':
                        tagColor = '#2ECC40';
                        tagText = t('completed');
                        break;
                }

                return <Tag color={tagColor}>{tagText}</Tag>;
            },
        },
        {
            title: t('common:created-at'),
            dataIndex: 'createdAt',
            key: 'createdAt',
            width: 200,
            render: (_: unknown, studyGroup) => {
                return dateTimeTransformer(studyGroup.createdAt);
            },
        },
    ] as TableColumnProps<StudyGroup>[];

    const breadCrumbItems = [
        {
            label: t('layout:study-group'),
            path: '/study-group',
        },
    ];

    return (
        <Layout
            member={member}
            breadCrumbItems={breadCrumbItems}
            seoConfig={{
                title: t('study-group'),
            }}
            withBackground
            activeMenu={['study-group']}
        >
            <div className="flex flex-col gap-2 sm:flex-row justify-between mb-4">
                <div className="flex">
                    <FilterDrawer
                        filterStudyGroupForm={filterStudyGroupForm}
                        onSearch={() => studyGroupListQuery.refetch()}
                        onReset={onResetFilterHandler}
                        loading={studyGroupListQuery.isLoading}
                    />
                    <Button type="link" className="list-none" onClick={onResetFilterHandler}>
                        {t('common:reset-filter')}
                    </Button>
                </div>
                <ExportData query={filterStudyGroupForm.getFieldsValue()} />
            </div>
            <div className="table_container">
                <Table
                    columns={columns}
                    dataSource={studyGroupListQuery.data}
                    loading={studyGroupListQuery.isLoading}
                    rowKey={(record) => record.id}
                    scroll={{ x: 1000 }}
                    onChange={paginationOnChange}
                    pagination={{
                        current: pagination.page,
                        pageSize: pagination.pageSize,
                        defaultPageSize: 1,
                        showSizeChanger: true,
                        pageSizeOptions: [10, 25, 50, 100],
                        showTotal: (total, range) => {
                            return t('common:pagination', { range0: range[0], range1: range[1], total: total });
                        },
                        total: pagination.total,
                    }}
                />
            </div>
        </Layout>
    );
};

export default StudyGroupPage;

export const getServerSideProps: GetServerSideProps = async ({ locale, req, resolvedUrl }) => {
    try {
        const authResponse = await authentication(req);

        return {
            props: {
                member: authResponse,
                ...(await serverSideTranslations(locale as string, [
                    'study-group',
                    'layout',
                    'common',
                    'messages',
                    'auth',
                    'api-messages',
                    'my-profile',
                ])),
            },
        };
    } catch (error: any) {
        if (error.response?.data?.unauthorized) {
            return {
                redirect: {
                    destination: localeLinkGenerator(locale, `/unauthorized`),
                    permanent: false,
                },
            };
        }

        return {
            redirect: {
                destination: localeLinkGenerator(locale, `/?redirect=/${resolvedUrl}`),
                permanent: false,
            },
        };
    }
};
