import React, { useEffect, useState } from 'react';
import PhoneInput from 'react-phone-input-2';
import 'react-phone-input-2/lib/style.css';
import { toast } from 'react-toastify';
import { GetServerSideProps, NextPage } from 'next';
import { useTranslation } from 'next-i18next';
import Link from 'next/link';
import { useRouter } from 'next/router';
import { useMutation, useQuery } from '@tanstack/react-query';
import { Button, Checkbox, Col, DatePicker, Divider, Form, Input, Modal, Row, Select } from 'antd';
import { register } from '@/services/registration';
import { AxiosErrorResponse, Member } from '@/types';
import { authentication, errorMessageFormatter, localeLinkGenerator } from '@/utils';
import Image from 'next/image';
import logo from '../../public/logo.png';
import { serverSideTranslations } from 'next-i18next/serverSideTranslations';
import HtmlParser from '@/components/HtmlParser';
import { getSupport } from '@/services/company-info';
import { getStates, getCities } from 'malaysia-postcodes';
  

const Register: NextPage = () => {
    const { t } = useTranslation(['register', 'messages', 'common']);
    const router = useRouter();
    const [registerForm] = Form.useForm();
    const [modal, setModal] = React.useState(false);
    const states = getStates();
    const [selectedState, setSelectedState] = useState<string | undefined>(undefined);
    const [cities, setCities] = useState<string[]>([]);

    useEffect(() => {
        if (selectedState) {
            registerForm.setFieldsValue({
                city: '',
            });

            if (selectedState === 'others') {
                return;
            }

            setCities(getCities(selectedState));
        }
    }, [selectedState, registerForm]);

    const registerMutation = useMutation({
        mutationFn: async (body: Member) => {
            const response = await register(body);
            return response;
        },
        onSettled: () => {
            registerForm.resetFields();
            router.push('/');
        },
    });

    const { data } = useQuery({
        queryKey: ['contact-information'],
        queryFn: async () => {
            const response = await getSupport();

            return response.data;
        },
    });

    const onRegisterHandler = () => {
        if (!registerForm.getFieldValue('termsAndCondition')) {
            return toast.error(t('messages:please-accept-terms-and-conditions'));
        }

        registerForm.validateFields().then(async (values) => {
            toast.promise(registerMutation.mutateAsync(values), {
                pending: t('messages:registering'),
                success: t('messages:successfully-registered'),
                error: {
                    render({ data }) {
                        return t(errorMessageFormatter(data as AxiosErrorResponse));
                    },
                },
            });
        });
    };

    const onCloseModal = () => {
        setModal(false);
    };

    return (
        <div className="flex items-center justify-center h-full loginBackground">
            <div className="w-full max-w-[600px] px-5 my-10">
                <div className="relative mb-6 w-full text-center md:px-10">
                    <Image src={logo} alt="Company Logo" style={{ width: '180px', height: 'auto' }} />
                </div>
                <div>
                    <Form form={registerForm} layout="vertical" initialValues={[{ country: 'Malaysia' }]}>
                        <Row gutter={[16, 0]}>
                            <Col span={24}>
                                <Form.Item label={t('full-name')} name="fullName" rules={[{ required: true, message: t('messages:required') }]}>
                                    <Input size="large" />
                                </Form.Item>
                            </Col>
                            <Col span={24}>
                                <Form.Item label={t('preferred-name')} name="preferredName">
                                    <Input size="large" />
                                </Form.Item>
                            </Col>
                            <Col span={24}>
                                <Form.Item label={t('email')} name="email" rules={[{ required: true, type: 'email' }]}>
                                    <Input size="large" />
                                </Form.Item>
                            </Col>
                            <Col xs={24} lg={12}>
                                <Form.Item label={t('phone-number')} name="phoneNumber" rules={[{ required: true }]}>
                                    <PhoneInput
                                        inputProps={{
                                            required: true,
                                        }}
                                        inputClass="form-control"
                                        country={'my'}
                                        inputStyle={{
                                            width: '100%',
                                            height: '40px',
                                            borderRadius: '6px',
                                        }}
                                        buttonStyle={{ borderRadius: '8px 0px 0px 8px' }}
                                    />
                                </Form.Item>
                            </Col>
                            <Col xs={24} lg={12}>
                                <Form.Item label={t('date-of-birth')} name="dateOfBirth" rules={[{ required: true }]}>
                                    <DatePicker className="!w-full" size="large" />
                                </Form.Item>
                            </Col>
                            <Col span={24}>
                                <Form.Item label={t('address')} name="address" rules={[{ required: true }]}>
                                    <Input size="large" />
                                </Form.Item>
                            </Col>
                            <Col xs={24} lg={12}>
                                <Form.Item label={t('state')} name="state" rules={[{ required: true }]}>
                                    <Select
                                        size="large"
                                        options={[
                                            ...states.map((state) => ({
                                                label: state,
                                                value: state,
                                            })),
                                            {
                                                label: "Others",
                                                value: "others",
                                            },
                                        ]}
                                        onChange={(value) => setSelectedState(value)}
                                    />
                                </Form.Item>
                            </Col>
                            <Col xs={24} lg={12}>
                                <Form.Item label={t('city')} name="city">
                                    <Select
                                        size="large"
                                        options={
                                            cities.map((city) => ({
                                                label: city,
                                                value: city,
                                            }))
                                        }
                                        disabled={!selectedState || selectedState === 'others'}
                                    />
                                </Form.Item>
                            </Col>
                            {
                                selectedState === 'others' && (
                                    <Col span={24}>
                                        <Form.Item label={t('country')} name="country" rules={[{ required: true }]}>
                                            <Input size="large" />
                                        </Form.Item>
                                    </Col>
                                )
                            }
                            <Col span={24}>
                                <Form.Item label={t('source-of-knowing-us')} name="source">
                                    <Input size="large" />
                                </Form.Item>
                            </Col>
                        </Row>
                        {/* Terms and Conditions */}
                        <h3 className="text-center text-[#313131]">{t('terms-and-conditions')}</h3>
                        <Form.Item name="termsAndCondition" rules={[{ required: true }]} valuePropName="checked">
                            <Checkbox className="mt-2">
                                <div className="text-center flex">
                                    <span>
                                        {`${t('by-clicking-register-you-agree-to-our-')}`}
                                        <span className="cursor-pointer text-blue-400" onClick={() => setModal(true)}>
                                            {`${t('terms-and-conditions')}.`}
                                        </span>
                                    </span>
                                </div>
                            </Checkbox>
                        </Form.Item>
                    </Form>

                    <Button type="primary" block onClick={onRegisterHandler} loading={registerMutation.isLoading}>
                        {t('common:register')}
                    </Button>
                </div>
                <Divider />
                <p className="text-center">
                    {t('already-have-an-account')} <Link href="/">{t('login-now')}</Link>
                </p>
            </div>
            <Modal
                title={t('terms-and-conditions')}
                open={modal}
                footer={null}
                onCancel={() => {
                    onCloseModal();
                }}
            >
                <HtmlParser html={data?.termsAndCondition} />
            </Modal>
        </div>
    );
};

export default Register;

export const getServerSideProps: GetServerSideProps = async ({ req, locale }) => {
    try {
        await authentication(req);

        return {
            redirect: {
                destination: localeLinkGenerator(locale, '/'),
                permanent: false,
                locale,
            },
        };
    } catch (error) {
        return {
            props: {
                ...(await serverSideTranslations(locale as string, ['register', 'common', 'messages', 'api-messages'])),
            },
        };
    }
};
