import { useStudyGroupContext } from '@/providers/StudyGroupContext';
import { Empty, Form, Tabs, Tag } from 'antd';
import { useTranslation } from 'next-i18next';
import { AxiosErrorResponse, Member, StudyGroupStatus } from '@/types';
import { errorMessageFormatter, mediaUrlGenerator } from '@/utils';
import { useEffect } from 'react';
import Layout from '../layouts';
import { getMemberPenalty } from '@/services/study-group';
import { useRouter } from 'next/router';
import { useQuery } from '@tanstack/react-query';
import { toast } from 'react-toastify';
import { currencyFormatter } from '@/utils/currencyFormatter';
import Task from './tabs/task';
import HtmlParser from '../HtmlParser';
import Link from 'next/link';
import { dateTransformer } from '@/utils/timeTransformer';

interface StudyGroupModuleProps {
    member: Member;
}

const StudyGroupModule: React.FC<StudyGroupModuleProps> = ({ member }) => {
    const { t } = useTranslation(['study-group', 'common', 'messages', 'layout']);
    const { studyGroupMember, studyGroup } = useStudyGroupContext();
    const router = useRouter();
    const { studyGroupId } = router.query;

    const studyGroupData = studyGroup.data;
    const studyGroupMemberData = studyGroupMember.data;

    const bookLink = studyGroupData?.book?.link;

    useEffect(() => {
        studyGroupDescriptionForm.setFieldsValue({
            description: studyGroupData?.description,
        });
    });

    // Query
    const { data: memberPenaltyData } = useQuery({
        queryKey: ['member-comment', studyGroupId],
        enabled: !!studyGroupId,
        queryFn: async () => {
            const response = await getMemberPenalty(studyGroupId as string);
            return response.data;
        },
        onError: (error: AxiosErrorResponse & Error) => {
            toast.error(t(errorMessageFormatter(error)));
        },
    });

    // Form Instance
    const [studyGroupDescriptionForm] = Form.useForm();
    const [studyGroupDataForm] = Form.useForm();

    // Function
    const renderGroupStatus = (status: StudyGroupStatus | undefined) => {
        switch (status) {
            case StudyGroupStatus.DRAFT:
                return (
                    <Tag color="#FFA500" className="w-fit">
                        {t('preparing')}
                    </Tag>
                );
            case StudyGroupStatus.GENERATED:
                return (
                    <Tag color="#FFA500" className="w-fit">
                        {t('preparing')}
                    </Tag>
                );
            case StudyGroupStatus.ONGOING:
                return (
                    <Tag color="#FFD700" className="w-fit">
                        {t('ongoing')}
                    </Tag>
                );
            case StudyGroupStatus.COMPLETED:
                return (
                    <Tag color="#2ECC40" className="w-fit">
                        {t('completed')}
                    </Tag>
                );
            default:
                return null;
        }
    };

    // Data
    const tabItems = [
        // Task Tab
        {
            label: t('task'),
            key: 'task',
            children: <Task />,
        },
    ];

    const breadCrumbItems = [
        {
            label: t('study-group'),
            path: '/study-group',
        },
        {
            label: studyGroupData?.name ?? t('common:no-data'),
            path: '/',
        },
    ];

    const seoConfig = {
        title: t('study-group'),
    };

    return (
        <Layout breadCrumbItems={breadCrumbItems} activeMenu={['study-group']} seoConfig={seoConfig} member={member} activeDropdown={['study-group']}>
            <div className="bg-white p-9 pb-4 mb-4 rounded-lg">
                <Form form={studyGroupDataForm}>
                    <p className="text-lg md:text-2xl text-[#544d4d] mb-2">
                        <b>{studyGroupData?.name}</b>
                    </p>
                    <p className="text-xs md:text-sm text-[#919191] w-fit">
                        <div className="grid grid-cols-1 sm:grid-cols-4 gap-2">
                            <span className="font-bold col-span-1">{t('total-members')}: </span>
                            <p className="col-span-3">{studyGroupMemberData?.memberCount}</p>
                            <span className="font-bold col-span-1">{t('period')}:</span>
                            <p className="col-span-3">
                                {dateTransformer(studyGroupData?.startDate ?? '')} - {dateTransformer(studyGroupData?.endDate ?? '')}
                            </p>
                            <span className="font-bold col-span-1">{t('total-penalties')}:</span>
                            <p className="col-span-3">MYR {currencyFormatter(memberPenaltyData?.totalAmount ?? '0')}</p>
                            <span className="font-bold col-span-1">{t('common:status')}:</span>
                            <p className="col-span-3">{renderGroupStatus(studyGroupData?.status)}</p>
                            <span className="font-bold col-span-1">{t('book-name')}:</span>
                            <p className="col-span-3">
                                <Link href={`/book/${studyGroupData?.book.id}`}>{studyGroupData?.book.name}</Link>
                            </p>
                            {studyGroupData?.book.uploadType === 'LINK' && <span className="font-bold col-span-1">{t('book-link')}:</span>}
                            {studyGroupData?.book.uploadType === 'FILE' && <span className="font-bold col-span-1">{t('book-file')}:</span>}

                            {studyGroupData?.book.uploadType === 'LINK' ? (
                                <div className="col-span-3">
                                    <a target="_blank" href={`${bookLink}`}>
                                        {bookLink}
                                    </a>
                                </div>
                            ) : (
                                studyGroupData?.book.uploadType === 'FILE' && (
                                    <div className="flex flex-col col-span-3">
                                        {studyGroupData?.book?.bookMedias.map((media) => {
                                            return (
                                                <a target="_blank" href={mediaUrlGenerator(media.media.key as string)} key={media.media.id}>
                                                    {media.media.name}
                                                </a>
                                            );
                                        })}
                                    </div>
                                )
                            )}

                            {studyGroupData?.description && (
                                <>
                                    <div className="font-bold col-span-1 text-[#919191]">{t('description')}:</div>
                                    <div className="col-span-3 text-[#919191] border-solid rounded-lg border-[#e8e8e8]">
                                        <div className="p-2 text-[#919191] text-xs md:text-sm">
                                            <HtmlParser html={studyGroupData?.description ?? ''} />
                                        </div>
                                    </div>
                                </>
                            )}
                        </div>
                    </p>
                </Form>
            </div>
            {!studyGroupData ? (
                <Empty />
            ) : (
                <div>
                    <Tabs items={tabItems} className="bg-white p-9 pt-5 rounded-lg" />
                </div>
            )}
        </Layout>
    );
};

export default StudyGroupModule;
