import { Button, Descriptions, Drawer, Tabs } from 'antd';
import { useTranslation } from 'next-i18next';
import CommentTab from '../tabs/Comment';
import React from 'react';
import dayjs from 'dayjs';
import { useStudyGroupContext } from '@/providers/StudyGroupContext';
import MedicalCertificate from './MedicalCertificate';
import { useQuery } from '@tanstack/react-query';
import { getMedicalCertificate } from '@/services/study-group';
import { AxiosErrorResponse } from '@/types';
import { toast } from 'react-toastify';
import { errorMessageFormatter } from '@/utils';
import HtmlParser from '@/components/HtmlParser';
import { dateTransformer } from '@/utils/timeTransformer';

const DItem = Descriptions.Item;

interface CommentsProps {
    open: boolean;
    setOpen: (open: boolean) => void;
}

const PreviewTask: React.FC<CommentsProps> = ({ open, setOpen }) => {
    const { t } = useTranslation(['study-group', 'layout', 'common', 'messages']);
    const { studyGroupTask, studyGroupTaskId } = useStudyGroupContext();
    const [medicalLeaveModal, setMedicalLeaveModal] = React.useState(false);

    const studyGroupTaskData = studyGroupTask.data;
    const isToday = dayjs(studyGroupTaskData?.taskDate).tz().isSame(dayjs.tz(), 'day');

    // Query
    const { data: medicalCertificateData, refetch: medicalCertificateRefetch } = useQuery({
        queryKey: ['medical-certificate', studyGroupTaskId],
        enabled: !!studyGroupTaskId,
        queryFn: async () => {
            const response = await getMedicalCertificate(studyGroupTaskId as string);
            return response.data;
        },
        onError: (error: AxiosErrorResponse & Error) => {
            toast.error(t(errorMessageFormatter(error)));
        },
    });

    // Data
    const tabItems = [
        // Comment Tab
        {
            label: t('comment'),
            key: 'comment',
            children: <CommentTab />,
        },
    ];

    return (
        <Drawer
            open={open}
            onClose={() => {
                setOpen(false);
            }}
            placement="bottom"
            height="90%"
        >
            <Descriptions bordered size="small" className="mb-4" column={1} labelStyle={{ maxWidth: '120px' }}>
                <DItem label={t('description')} contentStyle={{ fontWeight: 'normal' }}>
                    {studyGroupTaskData?.description ? (
                        <div className="text-justify max-h-[300px] overflow-auto">
                            <HtmlParser html={studyGroupTaskData.description} />
                        </div>
                    ) : (
                        '-'
                    )}
                </DItem>
                <DItem label={t('task-date')} contentStyle={{ fontWeight: 'normal' }}>
                    <div className="flex justify-between">{dateTransformer(studyGroupTaskData?.taskDate ?? '')}</div>
                </DItem>
            </Descriptions>
            {isToday && (
                <div className="w-full flex justify-end">
                    <Button
                        type="primary"
                        ghost
                        onClick={() => {
                            setMedicalLeaveModal(true);
                        }}
                        disabled={medicalCertificateData?.status === 'APPROVED' || medicalCertificateData?.status === 'PENDING'}
                        className={`${
                            medicalCertificateData?.status === 'REJECTED'
                                ? '!bg-red-500 !text-white'
                                : medicalCertificateData?.status === 'APPROVED'
                                  ? '!bg-green-500 !text-white'
                                  : ''
                        }`}
                    >
                        {medicalCertificateData?.status === 'PENDING'
                            ? t('applied')
                            : medicalCertificateData?.status === 'APPROVED'
                              ? t('common:approved')
                              : medicalCertificateData?.status === 'REJECTED'
                                ? t('common:rejected')
                                : t('apply-medical-leave')}
                    </Button>
                </div>
            )}

            <Tabs items={tabItems} className="bg-white pt-2 rounded-lg" />

            {/* Medical Leave */}
            <MedicalCertificate open={medicalLeaveModal} setOpen={setMedicalLeaveModal} medicalCertificateRefetch={medicalCertificateRefetch} />
        </Drawer>
    );
};

export default PreviewTask;
