import { useState } from 'react';
import 'react-phone-input-2/lib/style.css';
import { toast } from 'react-toastify';
import { useTranslation } from 'next-i18next';
import { UseQueryResult, useMutation } from '@tanstack/react-query';
import { Button, Form, Input, Modal } from 'antd';
import PasswordHintContainer from '@/components/ui/input/password/PasswordHintContainer';
import { updateMyPassword } from '@/services/my-profile';
import { AxiosErrorResponse, BaseModalProps, Member, UpdateMemberPasswordParams } from '@/types';
import { errorMessageFormatter } from '@/utils/errorFormatter';
import { passwordRegex } from '@/utils/regex';

interface PasswordModalProps extends BaseModalProps {
    memberQuery: UseQueryResult<Member | null>;
}

const PasswordModal: React.FC<PasswordModalProps> = ({ open, setOpen, memberQuery }) => {
    const { t } = useTranslation(['member', 'common']);
    const [form] = Form.useForm<{ password: string }>();
    const [password, setPassword] = useState<string>('');

    // Mutations
    const updateMyPasswordMutation = useMutation({
        mutationFn: async (body: UpdateMemberPasswordParams) => {
            const res = await updateMyPassword(body);
            return res.data;
        },
        onSuccess: () => {
            setOpen(false);
            memberQuery.refetch();
        },
    });

    // Functions
    const onCancelHandler = () => {
        form.resetFields();
        setOpen(false);
    };

    const onUpdateMemberPasswordHandler = () => {
        form.validateFields().then(async (values) => {
            toast.promise(updateMyPasswordMutation.mutateAsync(values), {
                pending: t('messages:updating-member-password'),
                success: t('messages:member-password-updated'),
                error: {
                    render({ data }) {
                        return t(errorMessageFormatter(data as AxiosErrorResponse));
                    },
                },
            });
        });
    };

    const handlePasswordOnChange = (e: React.ChangeEvent<HTMLInputElement>) => {
        e.preventDefault();
        setPassword(e.target.value);
    };

    return (
        <Modal open={open} onCancel={onCancelHandler} footer={null} title={t('edit-password')}>
            <Form layout="vertical" form={form} className="mt-4">
                <Form.Item
                    name="password"
                    label={t('password')}
                    rules={[
                        { required: true },
                        {
                            pattern: passwordRegex,
                            message: '',
                        },
                    ]}
                    className="mb-5"
                >
                    <Input.Password size="large" classNames={{ prefix: '!me-2' }} onChange={handlePasswordOnChange} />
                </Form.Item>
                <div className="mt-5 mb-5">
                    <PasswordHintContainer password={password} />
                </div>
                <Form.Item
                    name="confirmPassword"
                    label={t('confirm-password')}
                    dependencies={['password']}
                    rules={[
                        { required: true },
                        {
                            pattern: passwordRegex,
                            message: '',
                        },
                        {
                            validator: (_, value) => (value === password ? Promise.resolve() : Promise.reject(t('messages:password-not-match'))),
                        },
                    ]}
                >
                    <Input.Password size="large" classNames={{ prefix: '!me-2' }} />
                </Form.Item>
                <div className="flex gap-3">
                    <Button size="large" block onClick={onCancelHandler} loading={updateMyPasswordMutation.isLoading}>
                        {t('common:cancel')}
                    </Button>
                    <Button size="large" type="primary" block onClick={onUpdateMemberPasswordHandler} loading={updateMyPasswordMutation.isLoading}>
                        {t('common:update')}
                    </Button>
                </div>
            </Form>
        </Modal>
    );
};

export default PasswordModal;
