import { Button, Form, InputNumber, Modal, Select } from 'antd';
import { useTranslation } from 'next-i18next';
import { Dispatch, SetStateAction, useEffect } from 'react';
import { useMutation, useQuery, useQueryClient } from '@tanstack/react-query';
import { RequestBookTokenPayload } from '@/types/book-token';
import { bookTokenKeys, requestBookToken } from '@/services/book-token';
import { currencyFormatter, currencyParser, errorMessageFormatter } from '@/utils';
import { AxiosErrorResponse } from '@/types';
import { toast } from 'react-toastify';
import { getPackageList } from '@/services/data';

interface RequestBookTokenProps {
    open: boolean;
    setOpen: Dispatch<SetStateAction<boolean>>;
}

const RequestBookToken: React.FC<RequestBookTokenProps> = ({ open, setOpen }) => {
    const { t } = useTranslation(['book-token', 'common', 'messages', 'layout']);
    const [requestBookTokenForm] = Form.useForm<RequestBookTokenPayload>();
    const queryClient = useQueryClient();
    const packageSelected = Form.useWatch('selectedPackage', requestBookTokenForm);
    const packageUnitPrice = Form.useWatch('unitPrice', requestBookTokenForm);
    const quantity = Form.useWatch('quantity', requestBookTokenForm);

    const { data } = useQuery({
        queryKey: ['data', 'package'],
        queryFn: async () => {
            const res = await getPackageList();

            return res.data;
        },
    });

    useEffect(() => {
        if (packageSelected) {
            const selectedPackage = data?.find((item) => item.id === packageSelected);

            if (selectedPackage) {
                requestBookTokenForm.setFieldsValue({
                    unitPrice: selectedPackage.price,
                });
            }
        }
    }, [packageSelected]);

    useEffect(() => {
        if (quantity && packageUnitPrice) {
            requestBookTokenForm.setFieldsValue({
                totalAmount: quantity * packageUnitPrice,
            });
        }
    }, [quantity, packageUnitPrice]);

    const requestBookTokenMutation = useMutation({
        mutationFn: async (data: RequestBookTokenPayload) => {
            const res = await requestBookToken(data);
            return res.data;
        },
        onSuccess() {
            queryClient.invalidateQueries(bookTokenKeys.pagination());
            onCancelHandler();
        },
    });

    const onCancelHandler = () => {
        requestBookTokenForm.resetFields();
        setOpen(false);
    };

    const requestBookTokenHandler = () => {
        requestBookTokenForm.validateFields().then(async (values) => {
            toast.promise(requestBookTokenMutation.mutateAsync(values), {
                pending: t('messages:requesting-book-token'),
                success: t('messages:book-token-requested'),
                error: {
                    render({ data }) {
                        return t(errorMessageFormatter(data as AxiosErrorResponse));
                    },
                },
            });
        });
    };

    return (
        <Modal title={t('request-book-token')} footer={null} open={open} onCancel={onCancelHandler}>
            <Form form={requestBookTokenForm} layout="vertical" initialValues={{ package: '', unitPrice: 0, totalAmount: 0 }}>
                <Form.Item name="selectedPackage" label={t('package')} rules={[{ required: true }]}>
                    <Select>
                        <Select.Option key={''} value={''} disabled>
                            {t('common:please-select')}
                        </Select.Option>
                        {data?.map((item) => (
                            <Select.Option key={item.id} value={item.id}>
                                {item.name}
                            </Select.Option>
                        ))}
                    </Select>
                </Form.Item>
                <Form.Item name="unitPrice" label={t('unit-price')} rules={[{ required: true }]}>
                    <InputNumber formatter={currencyFormatter} parser={currencyParser} className="w-full" addonBefore={'MYR'} disabled />
                </Form.Item>
                <Form.Item name="quantity" label={t('quantity')} rules={[{ required: true }]}>
                    <InputNumber className="w-full" />
                </Form.Item>
                <Form.Item name="totalAmount" label={t('total-amount')} rules={[{ required: true }]}>
                    <InputNumber formatter={currencyFormatter} parser={currencyParser} className="w-full" addonBefore={'MYR'} disabled />
                </Form.Item>
            </Form>
            <div className="flex flex-row gap-4">
                <Button block onClick={onCancelHandler} loading={requestBookTokenMutation.isLoading}>
                    {t('common:cancel')}
                </Button>
                <Button block type="primary" onClick={requestBookTokenHandler} loading={requestBookTokenMutation.isLoading}>
                    {t('request-book-token')}
                </Button>
            </div>
        </Modal>
    );
};

export default RequestBookToken;
