import { HttpException, HttpStatus, Injectable } from '@nestjs/common';
import { Prisma, StudyBookRequestStatus } from '@prisma/client';
import { PrismaService } from 'src/prisma/prisma.service';
import { ApproveStudyRequestDto } from './study-request.dto';
import { StudyRequestQuery } from 'src/types/study-group';
import { conditionalReturn } from 'src/utils';
import * as dayjs from 'dayjs';

@Injectable()
export class StudyRequestService {
    constructor(private prisma: PrismaService) {}

    async getBookRequestByPagination(bookId: string, query: StudyRequestQuery) {
        const { page, pageSize, sortField, sortOrder, name, email, phoneNumber, createdAt, status } = query;

        const whereOptions: Prisma.StudyBookRequestWhereInput = {
            deletedAt: null,
            bookId,
            ...conditionalReturn(!!name || !!email || !!phoneNumber, {
                member: {
                    ...conditionalReturn(!!name, {
                        fullName: { mode: 'insensitive', contains: name },
                    }),
                    ...conditionalReturn(!!email, {
                        email: { mode: 'insensitive', contains: email },
                    }),
                    ...conditionalReturn(!!phoneNumber, {
                        phoneNumber: { mode: 'insensitive', contains: phoneNumber },
                    }),
                },
            }),
            ...conditionalReturn(createdAt && createdAt.length === 2, {
                createdAt: {
                    gte: createdAt && createdAt.length === 2 && new Date(dayjs(createdAt[0]).startOf('day').toISOString()),
                    lte: createdAt && createdAt.length === 2 && new Date(dayjs(createdAt[1]).endOf('day').toISOString()),
                },
            }),
            ...conditionalReturn(!!status, {
                status,
            }),
        };

        const bookRequestCount = await this.prisma.studyBookRequest.count({
            where: whereOptions,
        });

        const currentPage = this.prisma.pageCounter(bookRequestCount, page, pageSize);

        const bookRequestList = await this.prisma.studyBookRequest.findMany({
            where: whereOptions,
            skip: (currentPage - 1) * pageSize,
            take: pageSize,
            orderBy: {
                [!sortField ? 'createdAt' : sortField]: sortOrder ?? 'asc',
            },
            select: {
                ...this.prisma.createSelect(['status', 'createdAt']),
                member: {
                    select: {
                        ...this.prisma.createSelect(['id', 'fullName', 'preferredName', 'phoneNumber', 'email']),
                    },
                },
            },
        });

        return {
            count: bookRequestCount,
            rows: bookRequestList,
            page: currentPage,
        };
    }

    async getAllBookRequestByPagination(query: StudyRequestQuery) {
        const { page, pageSize, sortField, sortOrder, name, bookName, email, phoneNumber, createdAt, status } = query;

        const whereOptions: Prisma.StudyBookRequestWhereInput = {
            deletedAt: null,
            ...conditionalReturn(!!name, {
                member: {
                    fullName: { mode: 'insensitive', contains: name },
                },
            }),
            ...conditionalReturn(!!bookName, {
                book: {
                    name: { mode: 'insensitive', contains: bookName },
                },
            }),
            ...conditionalReturn(!!email, {
                member: {
                    email: { mode: 'insensitive', contains: email },
                },
            }),
            ...conditionalReturn(!!phoneNumber, {
                member: {
                    phoneNumber: { mode: 'insensitive', contains: phoneNumber },
                },
            }),
            ...conditionalReturn(createdAt && createdAt.length === 2, {
                createdAt: {
                    gte: createdAt && createdAt.length === 2 && new Date(dayjs(createdAt[0]).startOf('day').toISOString()),
                    lte: createdAt && createdAt.length === 2 && new Date(dayjs(createdAt[1]).endOf('day').toISOString()),
                },
            }),
            ...conditionalReturn(!!status, {
                status,
            }),
        };

        const bookRequestCount = await this.prisma.studyBookRequest.count({
            where: whereOptions,
        });

        const currentPage = this.prisma.pageCounter(bookRequestCount, page, pageSize);

        const bookRequestList = await this.prisma.studyBookRequest.findMany({
            where: whereOptions,
            skip: (currentPage - 1) * pageSize,
            take: pageSize,
            orderBy: {
                [!sortField ? 'createdAt' : sortField]: sortOrder ?? 'asc',
            },
            select: {
                ...this.prisma.createSelect(['id', 'status', 'createdAt', 'reason']),
                member: {
                    select: {
                        ...this.prisma.createSelect(['id', 'fullName', 'preferredName', 'phoneNumber', 'email']),
                    },
                },
                book: {
                    select: {
                        ...this.prisma.createSelect(['id', 'name']),
                    },
                },
            },
        });

        return {
            total: bookRequestCount,
            rows: bookRequestList,
            page: currentPage,
        };
    }

    async getStudyGroupList(bookId: string) {
        const bookRequestList = await this.prisma.studyGroup.findMany({
            where: {
                bookId,
                status: {
                    in: ['DRAFT', 'GENERATED'],
                },
                deletedAt: null,
            },
        });

        return bookRequestList;
    }

    async deleteStudyRequest(memberId: string, requestId: string) {
        const bookRequest = await this.prisma.studyBookRequest.findFirst({
            where: {
                id: requestId,
                memberId,
                deletedAt: null,
            },
        });

        if (!bookRequest) {
            throw new HttpException('api-messages:study-request-not-found', HttpStatus.NOT_FOUND);
        }

        const response = await this.prisma.studyBookRequest.update({
            where: {
                id: bookRequest.id,
            },
            data: {
                deletedAt: new Date(),
            },
        });

        return response;
    }

    async rejectStudyRequest(memberId: string, requestId: string, reason: string) {
        const bookRequest = await this.prisma.studyBookRequest.findFirst({
            where: {
                id: requestId,
                deletedAt: null,
            },
        });

        if (!bookRequest) {
            throw new HttpException('api-messages:study-request-not-found', HttpStatus.NOT_FOUND);
        }

        await this.prisma.member.update({
            where: {
                id: memberId,
            },
            data: {
                bookTokens: {
                    increment: 1,
                },
            },
        });

        const response = await this.prisma.studyBookRequest.update({
            where: {
                id: bookRequest.id,
            },
            data: {
                status: 'REJECTED',
                reason,
            },
        });

        return response;
    }

    async approveStudyRequest(bookId: string, memberId: string, values: ApproveStudyRequestDto) {
        const studyRequest = await this.prisma.studyBookRequest.findFirst({
            where: {
                bookId,
                memberId,
                deletedAt: null,
                status: StudyBookRequestStatus.PENDING,
            },
        });

        if (!studyRequest) {
            throw new HttpException('api-messages:study-request-not-found', HttpStatus.NOT_FOUND);
        }

        if (values.groupId) {
            await this.prisma.$transaction(async (tx) => {
                // Member already exist in study group
                const memberExist = await tx.studyGroupMember.findFirst({
                    where: {
                        memberId,
                        studyGroupId: values.groupId,
                    },
                });

                if (memberExist) {
                    throw new HttpException('api-messages:study-group-member-exist', HttpStatus.BAD_REQUEST);
                }

                // Update study book request
                const updatedStudyRequest = await tx.studyBookRequest.update({
                    where: {
                        id: studyRequest.id,
                    },
                    data: {
                        status: 'JOINED',
                    },
                });

                if (!updatedStudyRequest) {
                    throw new HttpException('api-messages:study-request-update-failed', HttpStatus.INTERNAL_SERVER_ERROR);
                }

                // Add to study group
                const studyGroupMember = await tx.studyGroupMember.create({
                    data: {
                        memberId,
                        studyGroupId: values.groupId,
                    },
                });

                if (!studyGroupMember) {
                    throw new HttpException('api-messages:study-group-member-create-failed', HttpStatus.INTERNAL_SERVER_ERROR);
                }
            });
        } else {
            await this.prisma.$transaction(async (tx) => {
                // Update study book request
                const studyBookRequest = await tx.studyBookRequest.update({
                    where: {
                        id: studyRequest.id,
                    },
                    data: {
                        status: StudyBookRequestStatus.JOINED,
                    },
                });

                if (!studyBookRequest) {
                    throw new HttpException('api-messages:study-request-update-failed', HttpStatus.INTERNAL_SERVER_ERROR);
                }

                // Create new study group
                const studyGroup = await tx.studyGroup.create({
                    data: {
                        bookId,
                        name: values.groupName,
                        startDate: values.groupPeriod[0],
                        endDate: values.groupPeriod[1],
                    },
                    select: this.prisma.createSelect(['id']),
                });

                if (!studyGroup) {
                    throw new HttpException('api-messages:study-group-create-failed', HttpStatus.INTERNAL_SERVER_ERROR);
                }

                // Add to study group
                const studyGroupMember = await tx.studyGroupMember.create({
                    data: {
                        memberId,
                        studyGroupId: studyGroup.id,
                    },
                });

                if (!studyGroupMember) {
                    throw new HttpException('api-messages:study-group-member-create-failed', HttpStatus.INTERNAL_SERVER_ERROR);
                }
            });
        }

        return studyRequest;
    }

    async getAllStudyRequest() {
        return await this.prisma.studyBookRequest.count({
            where: {
                deletedAt: null,
                status: StudyBookRequestStatus.PENDING,
            },
        });
    }
}
