import { Controller, Get, HttpStatus, Query, HttpCode, Param, Delete, Body, Put } from '@nestjs/common';
import { StaffInRequest, User } from 'src/types';
import { Authenticator, Authorizer, Staff } from 'src/auth/auth.decorator';
import { StudyRequestService } from './study-request.service';
import { JoiValidationPipe } from 'src/utils/validators/joi.validation.pipe';
import { StudyGroupQuery, StudyRequestQuery } from 'src/types/study-group';
import { studyRequestQuerySchema } from './study-request.validator';
import { ActivityLogService } from 'src/activity-log/activity-log.service';

@Controller()
export class StudyRequestController {
    constructor(
        private studyRequestService: StudyRequestService,
        private activityLogService: ActivityLogService,
    ) {}

    @Authenticator(User.STAFF)
    @Authorizer('STUDY_REQUEST_VIEW')
    @Get('all')
    @HttpCode(HttpStatus.OK)
    async getAllStudyRequestList(@Query(new JoiValidationPipe(studyRequestQuerySchema)) query: StudyRequestQuery) {
        const allStudyRequestList = await this.studyRequestService.getAllBookRequestByPagination(query);

        return allStudyRequestList;
    }

    @Authenticator(User.STAFF)
    @Authorizer('STUDY_REQUEST_VIEW')
    @Get(':bookId/single')
    @HttpCode(HttpStatus.OK)
    async getStudyRequestList(@Query(new JoiValidationPipe(studyRequestQuerySchema)) query: StudyRequestQuery, @Param('bookId') bookId: string) {
        const studyRequestList = await this.studyRequestService.getBookRequestByPagination(bookId, query);

        return studyRequestList;
    }

    @Authenticator(User.STAFF)
    @Authorizer('STUDY_REQUEST_VIEW')
    @Get(':bookId/study-group')
    @HttpCode(HttpStatus.OK)
    async getStudyGroupList(@Param('bookId') bookId: string) {
        const response = await this.studyRequestService.getStudyGroupList(bookId);

        return response;
    }

    @Authenticator(User.STAFF)
    @Authorizer('STUDY_REQUEST_UPDATE')
    @Delete(':memberId/:requestId/delete')
    @HttpCode(HttpStatus.OK)
    async deleteStudyRequest(@Staff() staff: StaffInRequest, @Param('requestId') requestId: string, @Param('memberId') memberId: string) {
        const { id: staffId, fullName } = staff;
        const response = await this.studyRequestService.deleteStudyRequest(memberId, requestId);

        await this.activityLogService.writeLogger({
            targetId: {
                tableName: 'study-request',
                id: response.id,
            },
            staffId,
            executorName: fullName,
            action: 'DELETE',
            description: 'api-messages:delete-study-request',
            data: response,
        });

        return response;
    }

    @Authenticator(User.STAFF)
    @Authorizer('STUDY_REQUEST_UPDATE')
    @Delete(':memberId/:requestId/reject')
    @HttpCode(HttpStatus.OK)
    async rejectStudyRequest(
        @Staff() staff: StaffInRequest,
        @Param('requestId') requestId: string,
        @Param('memberId') memberId: string,
        @Body('reason') reason: string,
    ) {
        const { id: staffId, fullName } = staff;
        const response = await this.studyRequestService.rejectStudyRequest(memberId, requestId, reason);

        await this.activityLogService.writeLogger({
            targetId: {
                tableName: 'study-request',
                id: response.id,
            },
            staffId,
            executorName: fullName,
            action: 'REJECT',
            description: 'api-messages:reject-study-request',
            data: response,
        });

        return response;
    }

    @Authenticator(User.STAFF)
    @Authorizer('STUDY_REQUEST_UPDATE')
    @Put(':bookId/:memberId/approve')
    @HttpCode(HttpStatus.OK)
    async approveStudyRequest(
        @Staff() staff: StaffInRequest,
        @Param('bookId') bookId: string,
        @Param('memberId') memberId: string,
        @Body() body: StudyGroupQuery,
    ) {
        const { id: staffId, fullName } = staff;
        const response = await this.studyRequestService.approveStudyRequest(bookId, memberId, body);

        await this.activityLogService.writeLogger({
            targetId: {
                tableName: 'study-request',
                id: response.id,
            },
            staffId,
            executorName: fullName,
            action: 'APPROVE',
            description: 'api-messages:approve-study-request',
            data: response,
        });

        return response;
    }
}
