import { Body, Controller, Delete, Get, HttpCode, HttpStatus, Param, Post, Put, Query } from '@nestjs/common';
import { Authenticator, Authorizer, Staff } from 'src/auth/auth.decorator';
import { StaffInRequest, User } from 'src/types';
import { JoiValidationPipe } from 'src/utils/validators/joi.validation.pipe';
import { StudyGroupService } from './study-group.service';
import {
    studyGroupQuerySchema,
    studyGroupCreateSchema,
    studyGroupDeleteSchema,
    studyGroupByIdSchema,
    studyGroupUpdateSchema,
    studyGroupTaskByIdSchema,
    studyGroupSubTaskCommentParamsSchema,
    studyGroupTaskDescriptionUpdateSchema,
    studyGroupMemberSchema,
    studyGroupMemberCreateSchema,
    studyGroupMembersSchema,
    studyGroupMemberDeleteSchema,
    studyGroupTaskCommentFilterSchema,
    exportStudyGroupMemberSchema,
    updateReplySchema,
} from './study-group.validator';
import { StudyGroupMembers, StudyGroupTaskCommentFilter } from 'src/types/study-group';
import { Member } from '@prisma/client';
import { ActivityLogService } from 'src/activity-log/activity-log.service';
import {
    CreateStudyGroupDto,
    CreateStudyGroupMemberParams,
    DeleteStudyGroupMemberDto,
    ExportStudyGroupMembers,
    StudyGroupFilterParams,
    StudyGroupParams,
    StudyGroupSubTaskCommentParams,
    StudyGroupTaskParams,
    UpdateDescriptionDto,
    UpdateReply,
    UpdateStudyGroupDto,
} from './study-group.dto';

@Controller()
export class StudyGroupController {
    constructor(
        private studyGroup: StudyGroupService,
        private activityLogService: ActivityLogService,
    ) {}

    // Context Provider
    @Authenticator(User.STAFF)
    @Authorizer('STUDY_GROUP_VIEW')
    @Get(':studyGroupId/id')
    @HttpCode(HttpStatus.OK)
    async getStudyGroupById(@Param(new JoiValidationPipe(studyGroupByIdSchema)) param: StudyGroupParams) {
        const studyGroup = await this.studyGroup.getStudyGroupById(param.studyGroupId);

        return studyGroup;
    }

    @Authenticator(User.STAFF)
    @Authorizer('STUDY_GROUP_TASK_VIEW')
    @Get(':studyGroupId/member')
    @HttpCode(HttpStatus.OK)
    async getStudyGroupMembers(@Param(new JoiValidationPipe(studyGroupByIdSchema)) param: StudyGroupParams) {
        const studyGroupMembers = await this.studyGroup.getStudyGroupMembers(param.studyGroupId);

        return studyGroupMembers;
    }

    @Authenticator(User.STAFF)
    @Authorizer('STUDY_GROUP_TASK_VIEW')
    @Get(':studyGroupTaskId/task-by-id')
    @HttpCode(HttpStatus.OK)
    async getStudyGroupTaskById(@Param(new JoiValidationPipe(studyGroupTaskByIdSchema)) param: StudyGroupTaskParams) {
        const studyGroupTask = await this.studyGroup.getStudyGroupTaskById(param.studyGroupTaskId);

        return studyGroupTask;
    }

    ///////////////////
    // Study Group
    ///////////////////

    // Get
    @Authenticator(User.STAFF)
    @Authorizer('STUDY_GROUP_VIEW')
    @Get()
    @HttpCode(HttpStatus.OK)
    async getStudyGroupList(@Query(new JoiValidationPipe(studyGroupQuerySchema)) query: StudyGroupFilterParams) {
        const studyGroupList = await this.studyGroup.getAllStudyGroupByPagination(query);

        return studyGroupList;
    }

    @Authenticator(User.STAFF)
    @Authorizer('STUDY_GROUP_VIEW')
    @Get('books')
    @HttpCode(HttpStatus.OK)
    async getBooks() {
        const books = await this.studyGroup.getBooks();

        return books;
    }

    @Authenticator(User.STAFF)
    @Authorizer('STUDY_GROUP_VIEW')
    @Get(':studyGroupId/study-group-penalty')
    @HttpCode(HttpStatus.OK)
    async getStudyGroupPenalty(@Param(new JoiValidationPipe(studyGroupByIdSchema)) param: StudyGroupParams) {
        const studyGroupPenalty = await this.studyGroup.getStudyGroupPenalty(param.studyGroupId);

        return studyGroupPenalty;
    }

    @Authenticator(User.STAFF)
    @Authorizer('STUDY_GROUP_VIEW')
    @Get(':studyGroupId/calculate-penalties')
    @HttpCode(HttpStatus.OK)
    async calculateAllPenaltiesByStudyGroup(@Param(new JoiValidationPipe(studyGroupByIdSchema)) param: StudyGroupParams) {
        const response = await this.studyGroup.calculateAllPenaltiesByStudyGroup(param.studyGroupId);

        return response;
    }

    @Authenticator(User.STAFF)
    @Authorizer('STUDY_GROUP_VIEW')
    @Get(':studyGroupId/member-list')
    @HttpCode(HttpStatus.OK)
    async getMemberListByPagination(
        @Query(new JoiValidationPipe(studyGroupMemberSchema)) query: Member,
        @Param(new JoiValidationPipe(studyGroupByIdSchema)) param: StudyGroupParams,
    ) {
        const memberList = await this.studyGroup.getMemberListByPagination(query, param.studyGroupId);

        return {
            total: memberList.count,
            rows: memberList.rows,
            page: memberList.page,
        };
    }

    @Authenticator(User.STAFF)
    @Authorizer('STUDY_GROUP_VIEW')
    @Get(':studyGroupId/members')
    @HttpCode(HttpStatus.OK)
    async getStudyGroupMemberList(
        @Query(new JoiValidationPipe(studyGroupMembersSchema)) query: StudyGroupMembers,
        @Param(new JoiValidationPipe(studyGroupByIdSchema)) param: StudyGroupParams,
    ) {
        const memberList = await this.studyGroup.getStudyGroupMemberList(param.studyGroupId, query);

        return {
            total: memberList.total,
            rows: memberList.rows,
            page: memberList.page,
        };
    }

    @Authenticator(User.STAFF)
    @Authorizer('BOOK_TOKEN_VIEW')
    @Get('/member-export/:studyGroupId')
    async getExportBookTokenSubscriptionList(
        @Query(new JoiValidationPipe(exportStudyGroupMemberSchema)) query: ExportStudyGroupMembers,
        @Param(new JoiValidationPipe(studyGroupByIdSchema)) param: StudyGroupParams,
    ) {
        const response = this.studyGroup.getExportStudyGroupMemberList(query, param.studyGroupId);
        return response;
    }

    // Add
    @Authenticator(User.STAFF)
    @Authorizer('STUDY_GROUP_CREATE')
    @Post('')
    @HttpCode(HttpStatus.OK)
    async addGroup(@Staff() staff: StaffInRequest, @Body(new JoiValidationPipe(studyGroupCreateSchema)) body: CreateStudyGroupDto) {
        const { id: staffId, fullName } = staff;
        const studyGroup = await this.studyGroup.createStudyGroup(body);

        await this.activityLogService.writeLogger({
            targetId: {
                tableName: 'study-group',
                id: studyGroup.id,
            },
            staffId,
            executorName: fullName,
            action: 'CREATE',
            description: 'api-messages:create-study-group',
            data: studyGroup,
        });

        return studyGroup;
    }

    @Authenticator(User.STAFF)
    @Authorizer('STUDY_GROUP_CREATE')
    @Post(':studyGroupId/member/:memberId')
    @HttpCode(HttpStatus.OK)
    async addMember(@Staff() staff: StaffInRequest, @Param(new JoiValidationPipe(studyGroupMemberCreateSchema)) param: CreateStudyGroupMemberParams) {
        const { id: staffId, fullName } = staff;
        const studyGroupMember = await this.studyGroup.createStudyGroupMember(param.studyGroupId, param.memberId);

        await this.activityLogService.writeLogger({
            targetId: {
                tableName: 'study-group',
                id: studyGroupMember.studyGroupId,
            },
            staffId,
            executorName: fullName,
            action: 'UPDATE',
            description: 'api-messages:add-study-group-member',
            data: studyGroupMember,
        });

        return studyGroupMember;
    }

    // Update
    @Authenticator(User.STAFF)
    @Authorizer('STUDY_GROUP_UPDATE')
    @Put(':studyGroupId')
    @HttpCode(HttpStatus.OK)
    async updateStudyGroupById(
        @Staff() staff: StaffInRequest,
        @Param(new JoiValidationPipe(studyGroupByIdSchema)) param: StudyGroupParams,
        @Body(new JoiValidationPipe(studyGroupUpdateSchema)) body: UpdateStudyGroupDto,
    ) {
        const { id: staffId, fullName } = staff;
        const studyGroup = await this.studyGroup.updateStudyGroupById(param.studyGroupId, body);

        await this.activityLogService.writeLogger({
            targetId: {
                tableName: 'study-group',
                id: studyGroup.id,
            },
            staffId,
            executorName: fullName,
            action: 'UPDATE',
            description: 'api-messages:update-study-group',
            data: studyGroup,
        });

        return studyGroup;
    }

    // Delete
    @Authenticator(User.STAFF)
    @Authorizer('STUDY_GROUP_DELETE')
    @Delete(':studyGroupId')
    @HttpCode(HttpStatus.OK)
    async deleteStudyGroupById(@Staff() staff: StaffInRequest, @Param(new JoiValidationPipe(studyGroupDeleteSchema)) param: StudyGroupParams) {
        const { id: staffId, fullName } = staff;
        const studyGroup = await this.studyGroup.deleteStudyGroupById(param.studyGroupId);

        await this.activityLogService.writeLogger({
            targetId: {
                tableName: 'study-group',
                id: studyGroup[0].id,
            },
            staffId,
            executorName: fullName,
            action: 'DELETE',
            description: 'api-messages:delete-study-group',
            data: studyGroup[0],
        });

        return studyGroup;
    }

    @Authenticator(User.STAFF)
    @Authorizer('STUDY_GROUP_DELETE')
    @Delete(':studyGroupId/member/:memberId')
    @HttpCode(HttpStatus.OK)
    async deleteStudyGroupMemberById(
        @Staff() staff: StaffInRequest,
        @Param(new JoiValidationPipe(studyGroupMemberDeleteSchema)) param: DeleteStudyGroupMemberDto,
    ) {
        const { id: staffId, fullName } = staff;
        const studyGroup = await this.studyGroup.deleteStudyGroupMemberById(param.studyGroupId, param.memberId);

        await this.activityLogService.writeLogger({
            targetId: {
                tableName: 'study-group',
                id: studyGroup.studyGroupId,
            },
            staffId,
            executorName: fullName,
            action: 'DELETE',
            description: 'api-messages:delete-study-group-member',
            data: studyGroup[0],
        });

        return studyGroup;
    }

    ///////////////////
    // Study Group Task
    ///////////////////

    // Get
    @Authenticator(User.STAFF)
    @Authorizer('STUDY_GROUP_TASK_VIEW')
    @Get(':studyGroupTaskId/task-member-penalty')
    @HttpCode(HttpStatus.OK)
    async getStudyGroupTaskMemberPenalty(@Param(new JoiValidationPipe(studyGroupTaskByIdSchema)) param: StudyGroupTaskParams) {
        const studyGroupTaskMemberPenalty = await this.studyGroup.getTaskMemberPenalty(param.studyGroupTaskId);

        return studyGroupTaskMemberPenalty;
    }

    @Authenticator(User.STAFF)
    @Authorizer('STUDY_GROUP_TASK_VIEW')
    @Get(':studyGroupTaskId/task-penalty')
    @HttpCode(HttpStatus.OK)
    async getTaskPenalty(@Param(new JoiValidationPipe(studyGroupTaskByIdSchema)) param: StudyGroupTaskParams) {
        const studyGroupTaskPenalty = await this.studyGroup.getTaskPenalty(param.studyGroupTaskId);

        return studyGroupTaskPenalty;
    }

    // Add
    @Authenticator(User.STAFF)
    @Authorizer('STUDY_GROUP_TASK_CREATE')
    @Post(':studyGroupId/generate-task')
    @HttpCode(HttpStatus.OK)
    async generateStudyGroupTask(@Staff() staff: StaffInRequest, @Param(new JoiValidationPipe(studyGroupByIdSchema)) param: StudyGroupParams) {
        const { id: staffId, fullName } = staff;
        const studyGroup = await this.studyGroup.generateStudyGroupTask(param.studyGroupId);

        await this.activityLogService.writeLogger({
            targetId: {
                tableName: 'study-group',
                id: studyGroup.id,
            },
            staffId,
            executorName: fullName,
            action: 'UPDATE',
            description: 'api-messages:generate-study-group-task',
            data: studyGroup,
        });

        return studyGroup;
    }

    // Update
    @Authenticator(User.STAFF)
    @Authorizer('STUDY_GROUP_TASK_UPDATE')
    @Put(':studyGroupTaskId/description')
    @HttpCode(HttpStatus.OK)
    async updateStudyGroupTaskDescription(
        @Staff() staff: StaffInRequest,
        @Param(new JoiValidationPipe(studyGroupTaskByIdSchema)) param: StudyGroupTaskParams,
        @Body(new JoiValidationPipe(studyGroupTaskDescriptionUpdateSchema)) body: UpdateDescriptionDto,
    ) {
        const { id: staffId, fullName } = staff;
        const studyGroupTask = await this.studyGroup.updateDescription(param.studyGroupTaskId, body);

        await this.activityLogService.writeLogger({
            targetId: {
                tableName: 'study-group',
                id: studyGroupTask.studyGroupId,
            },
            staffId,
            executorName: fullName,
            action: 'UPDATE',
            description: 'api-messages:update-study-group-task-description',
            data: studyGroupTask,
        });

        return studyGroupTask;
    }

    ///////////////////
    // Comment
    ///////////////////

    // Get
    @Authenticator(User.STAFF)
    @Authorizer('STUDY_GROUP_TASK_VIEW')
    @Get(':studyGroupTaskId/task-comments-id')
    @HttpCode(HttpStatus.OK)
    async getCommentsByTaskId(@Param(new JoiValidationPipe(studyGroupTaskByIdSchema)) param: StudyGroupTaskParams) {
        const taskComments = await this.studyGroup.getCommentsByTaskId(param.studyGroupTaskId);

        return taskComments;
    }

    @Authenticator(User.STAFF)
    @Authorizer('STUDY_GROUP_TASK_VIEW')
    @Get(':studyGroupId/group-comments')
    @HttpCode(HttpStatus.OK)
    async getCommentsByGroupId(@Param(new JoiValidationPipe(studyGroupByIdSchema)) param: StudyGroupParams) {
        const groupComments = await this.studyGroup.getCommentsByGroupId(param.studyGroupId);

        return groupComments;
    }

    @Authenticator(User.STAFF)
    @Authorizer('STUDY_GROUP_TASK_VIEW')
    @Get(':studyGroupTaskId/task-comments')
    @HttpCode(HttpStatus.OK)
    async getStudyGroupTaskComment(
        @Param(new JoiValidationPipe(studyGroupTaskByIdSchema)) param: StudyGroupTaskParams,
        @Query(new JoiValidationPipe(studyGroupTaskCommentFilterSchema)) query: StudyGroupTaskCommentFilter,
    ) {
        const studyGroupTaskComment = await this.studyGroup.getTaskComments(param.studyGroupTaskId, query);

        return studyGroupTaskComment;
    }

    @Authenticator(User.STAFF)
    @Get(':studyGroupTaskId/like')
    @HttpCode(HttpStatus.OK)
    async getTaskCommentLikes(@Param('studyGroupTaskId') studyGroupTaskId: string) {
        const taskCommentLikes = await this.studyGroup.getTaskCommentLikes(studyGroupTaskId);

        return taskCommentLikes;
    }

    @Authenticator(User.STAFF)
    @Get(':studyGroupTaskId/reply/:commentId')
    @HttpCode(HttpStatus.OK)
    async getTaskCommentReplies(@Param('studyGroupTaskId') studyGroupTaskId: string, @Param('commentId') commentId: string) {
        const taskCommentReplies = await this.studyGroup.getReplies(studyGroupTaskId, commentId);

        return taskCommentReplies;
    }

    @Authenticator(User.STAFF)
    @Get(':studyGroupTaskId/own-task-comment')
    @HttpCode(HttpStatus.OK)
    async getOwnTaskComment(@Param(new JoiValidationPipe(studyGroupTaskByIdSchema)) param: StudyGroupTaskParams, @Staff() staff: StaffInRequest) {
        const ownTaskComment = await this.studyGroup.getOwnTaskComment(param.studyGroupTaskId, staff.id);

        return ownTaskComment;
    }

    // Delete
    @Authenticator(User.STAFF)
    @Authorizer('STUDY_GROUP_TASK_DELETE')
    @Delete(':commentId/comment')
    @HttpCode(HttpStatus.OK)
    async deleteCommentById(
        @Staff() staff: StaffInRequest,
        @Param(new JoiValidationPipe(studyGroupSubTaskCommentParamsSchema)) param: StudyGroupSubTaskCommentParams,
    ) {
        const { id: staffId, fullName } = staff;
        const studyGroupComment = await this.studyGroup.deleteStudyGroupSubTaskComment(param.commentId);

        await this.activityLogService.writeLogger({
            targetId: {
                tableName: 'study-group',
                id: studyGroupComment.studyGroupTask.id,
            },
            staffId,
            executorName: fullName,
            action: 'DELETE',
            description: 'api-messages:delete-study-group-sub-task-comment',
            data: studyGroupComment,
        });

        return studyGroupComment;
    }

    // Update
    @Authenticator(User.STAFF)
    @Post(':studyGroupTaskId/like/:studyGroupTaskCommentId')
    @HttpCode(HttpStatus.OK)
    async likeTaskComment(
        @Param('studyGroupTaskId') studyGroupTaskId: string,
        @Staff() staff: StaffInRequest,
        @Param('studyGroupTaskCommentId') studyGroupTaskCommentId: string,
    ) {
        const likedComment = await this.studyGroup.likeTaskComment(studyGroupTaskCommentId, staff.id, studyGroupTaskId);

        return likedComment;
    }

    @Authenticator(User.STAFF)
    @Put(':studyGroupTaskId/reply')
    @HttpCode(HttpStatus.OK)
    async updateReply(
        @Param(new JoiValidationPipe(studyGroupTaskByIdSchema)) param: StudyGroupTaskParams,
        @Body(new JoiValidationPipe(updateReplySchema)) body: UpdateReply,
        @Staff() staff: StaffInRequest,
    ) {
        const updatedReply = await this.studyGroup.updateReply(param.studyGroupTaskId, staff?.id, body?.commentId, body?.reply);
        return updatedReply;
    }
}
