import { Body, Controller, Delete, Get, Param, Post, Put, Query, UploadedFile, UseInterceptors } from '@nestjs/common';
import { Authenticator, Authorizer, Staff } from 'src/auth/auth.decorator';
import { StaffInRequest, UploadType, User } from 'src/types';
import { BookService } from './book.service';
import { FileInterceptor } from '@nestjs/platform-express';
import { addBookSchema, bookIdParamSchema, bookQuerySchema, updateBookSchema } from './book.validator';
import { JoiValidationPipe } from 'src/utils/validators/joi.validation.pipe';
import { AddBookDto, BookQuery, UpdateBookDto, BookIdParam } from './book.dto';
import { ActivityLogService } from 'src/activity-log/activity-log.service';

@Controller()
export class BookController {
    constructor(
        private bookService: BookService,
        private activityLogService: ActivityLogService,
    ) {}

    @Authenticator(User.STAFF)
    @Authorizer('BOOK_VIEW')
    @Get('')
    async getBookByPagination(@Query(new JoiValidationPipe(bookQuerySchema)) query: BookQuery) {
        const bookList = await this.bookService.getBookByPagination(query);

        return {
            total: bookList.count,
            rows: bookList.rows,
            page: bookList.page,
        };
    }

    @Authenticator(User.STAFF)
    @Authorizer('BOOK_CREATE')
    @Post('add')
    async addBook(@Staff() staff: StaffInRequest, @Body(new JoiValidationPipe(addBookSchema)) body: AddBookDto) {
        const { id: staffId, fullName } = staff;

        const newBody = {
            ...body,
            ...(body.uploadType === UploadType.LINK && { link: this.checkHttpProtocol(body.link) }),
        };
        const response = await this.bookService.addBook(newBody);

        await this.activityLogService.writeLogger({
            targetId: {
                tableName: 'book',
                id: response.id,
            },
            staffId,
            executorName: fullName,
            action: 'CREATE',
            description: 'api-messages:add-book',
            data: response,
        });

        return response;
    }

    @Authenticator(User.STAFF)
    @Authorizer('BOOK_CREATE')
    @Post('upload')
    @UseInterceptors(FileInterceptor('file'))
    async uploadBookFile(@UploadedFile() bookFile: Express.Multer.File) {
        const response = await this.bookService.uploadBookFile(bookFile);

        return response;
    }

    @Authenticator(User.STAFF)
    @Authorizer('BOOK_VIEW')
    @Get(':bookId')
    async getBookById(@Param(new JoiValidationPipe(bookIdParamSchema)) param: BookIdParam) {
        const response = await this.bookService.getBookById(param.bookId);

        return response;
    }

    @Authenticator(User.STAFF)
    @Authorizer('BOOK_UPDATE')
    @Put(':bookId')
    async updateBookById(
        @Staff() staff: StaffInRequest,
        @Param(new JoiValidationPipe(bookIdParamSchema)) param: BookIdParam,
        @Body(new JoiValidationPipe(updateBookSchema)) body: UpdateBookDto,
    ) {
        const { id: staffId, fullName } = staff;
        const newBody = {
            ...body,
            ...(body.uploadType === UploadType.LINK && { link: this.checkHttpProtocol(body.link) }),
        };
        const response = await this.bookService.updateBookById(param.bookId, newBody);

        await this.activityLogService.writeLogger({
            targetId: {
                tableName: 'book',
                id: response.id,
            },
            staffId,
            executorName: fullName,
            action: 'UPDATE',
            description: 'api-messages:update-book',
            data: response,
        });

        return response;
    }

    @Authenticator(User.STAFF)
    @Authorizer('BOOK_UPDATE')
    @Put(':bookId/status')
    async updateBookStatus(@Param(new JoiValidationPipe(bookIdParamSchema)) param: BookIdParam, @Staff() staff: StaffInRequest) {
        const { id: staffId, fullName } = staff;
        const response = await this.bookService.updateBookStatus(param.bookId);

        await this.activityLogService.writeLogger({
            targetId: {
                tableName: 'book',
                id: response.id,
            },
            staffId,
            executorName: fullName,
            action: 'UPDATE',
            description: 'api-messages:update-book-status',
            data: response,
        });

        return response;
    }

    @Authenticator(User.STAFF)
    @Authorizer('BOOK_DELETE')
    @Delete(':bookId')
    async deleteBookById(@Staff() staff: StaffInRequest, @Param(new JoiValidationPipe(bookIdParamSchema)) param: BookIdParam) {
        const { id: staffId, fullName } = staff;
        const response = await this.bookService.deleteBookById(param.bookId);

        await this.activityLogService.writeLogger({
            targetId: {
                tableName: 'book',
                id: response.id,
            },
            staffId,
            executorName: fullName,
            action: 'DELETE',
            description: 'api-messages:delete-book',
            data: response,
        });

        return response;
    }

    private checkHttpProtocol = (url: string) => {
        const regex = /^(http|https):\/\//;

        if (!regex.test(url)) {
            // If 'http://' or 'https://' is not present, add 'http://'
            return 'https://' + url;
        }

        return url;
    };
}
