import { ForbiddenException, HttpException, HttpStatus, Injectable } from '@nestjs/common';
import { ConfigService } from '@nestjs/config';
import { JwtService } from '@nestjs/jwt';
import * as argon from 'argon2';
import * as dayjs from 'dayjs';
import roleData from 'src/data/role';
import { StaffJwtPayload } from 'src/types';
import { CustomException } from 'src/utils/CustomException';
import { v4 as uuidv4 } from 'uuid';
import { PrismaService } from '../../prisma/prisma.service';
import { EmailService } from 'src/email/email.service';
import { StaffLoginDto, VerifyStaffDto } from './auth.dto';
import { AccountStatus } from '@prisma/client';

@Injectable()
export class AuthService {
    constructor(
        private prisma: PrismaService,
        private jwtService: JwtService,
        private config: ConfigService,
        private emailService: EmailService,
    ) {}

    async authenticate(token: string, permission?: string) {
        const decoded: StaffJwtPayload = this.jwtService.verify(token, {
            secret: this.config.get('TOKEN_SECRET'),
        });

        // Get the staff details
        const staff = await this.prisma.staff.findUnique({
            where: {
                id: decoded.id,
            },
            select: {
                id: true,
                email: true,
                fullName: true,
                status: true,
                role: {
                    select: this.prisma.createSelect(roleData.exclude(['createdAt', 'updatedAt', 'deletedAt', 'superAdmin'])),
                },
            },
        });

        // If staff is not exist or the staff is deleted
        if (!staff) {
            throw new HttpException('api-messages:staff-not-found', HttpStatus.NOT_FOUND);
        }

        // Update last active
        await this.prisma.staff.update({
            where: {
                id: decoded.id,
            },
            data: {
                lastActiveAt: dayjs().toDate(),
            },
        });

        // If the staff is not active
        if (staff.status !== 'ACTIVE') {
            throw new HttpException('api-messages:staff-disabled', HttpStatus.UNAUTHORIZED);
        }

        // If the permission is valid
        if (permission !== '' && permission !== undefined && permission !== null) {
            const staff = await this.prisma.staff.findFirst({
                where: {
                    id: decoded.id,
                    deletedAt: null,
                },
                select: {
                    id: true,
                    role: {
                        select: {
                            [permission]: true,
                        },
                    },
                },
            });

            if (!staff) {
                return { unauthorized: true };
            }

            // If the permission is false
            if (!(staff.role as unknown as { [key: string]: boolean })[permission]) {
                throw new CustomException('api-messages:permission-denied', HttpStatus.UNAUTHORIZED, { unauthorized: true });
            }
        }

        return staff;
    }

    async login(loginBody: StaffLoginDto) {
        const staff = await this.prisma.staff.findFirst({
            where: {
                email: loginBody.email,
                deletedAt: null,
            },
            select: {
                id: true,
                email: true,
                password: true,
                fullName: true,
            },
        });

        // Check if staff exists
        if (!staff) {
            throw new ForbiddenException('api-messages:incorrect-staff-email-or-password');
        }

        // Check if staff is verified
        if (!staff.password) {
            throw new CustomException('api-messages:account-not-verified', HttpStatus.CONFLICT, {
                unverified: true,
            });
        }

        // Check if password is correct
        const isMatch = await argon.verify(staff.password, loginBody.password);

        if (!isMatch) {
            throw new ForbiddenException('api-messages:incorrect-staff-email-or-password');
        }

        const payload = {
            id: staff.id,
            email: staff.email,
            fullName: staff.fullName,
        };

        const staffToken = await this.signToken(payload);

        return staffToken;
    }

    async tokenVerifier(staffId: string, token: string) {
        const staff = await this.prisma.staff.findUnique({
            where: {
                id: staffId,
                deletedAt: null,
            },
            select: {
                id: true,
                email: true,
                fullName: true,
                tokens: {
                    select: {
                        token: true,
                        type: true,
                        expiredAt: true,
                    },
                    where: {
                        type: 'CONFIRMATION',
                        token,
                        usedAt: null,
                    },
                },
            },
        });

        if (!staff) {
            throw new HttpException('api-messages:staff-not-found', HttpStatus.NOT_FOUND);
        }

        if (staff.tokens.length === 0) {
            throw new HttpException('api-messages:invalid-token', HttpStatus.NOT_FOUND);
        }

        if (dayjs().isAfter(dayjs(staff.tokens[0].expiredAt))) {
            throw new CustomException('api-messages:token-expired', HttpStatus.BAD_REQUEST, {
                verificationExpired: true,
                email: staff.email,
            });
        }

        return staff;
    }

    async verifyStaff(body: VerifyStaffDto) {
        const { id, password, token } = body;

        const staff = await this.prisma.staff.findFirst({
            where: {
                id,
                deletedAt: null,
            },
            select: {
                id: true,
                fullName: true,
                email: true,
                tokens: {
                    select: {
                        token: true,
                        expiredAt: true,
                    },
                    where: {
                        type: 'CONFIRMATION',
                        token,
                        usedAt: null,
                    },
                },
            },
        });

        if (!staff) {
            throw new HttpException('api-messages:staff-not-found', HttpStatus.NOT_FOUND);
        }

        if (staff.tokens.length === 0) {
            throw new HttpException('api-messages:invalid-token', HttpStatus.UNAUTHORIZED);
        }

        if (dayjs().isAfter(dayjs(staff.tokens[0].expiredAt))) {
            throw new CustomException('api-messages:token-expired', HttpStatus.BAD_REQUEST, {
                verificationExpired: true,
                email: staff.email,
            });
        }

        // Hash password
        const hashedPassword = await argon.hash(password);

        // Update staff
        const updatedStaff = await this.prisma.staff.update({
            where: {
                id: staff.id,
            },
            data: {
                password: hashedPassword,
                tokens: {
                    update: {
                        where: {
                            token,
                        },
                        data: {
                            usedAt: dayjs().toDate(),
                        },
                    },
                },
                status: AccountStatus.ACTIVE,
            },
            select: {
                id: true,
                email: true,
                fullName: true,
            },
        });

        return updatedStaff;
    }

    async forgotPassword(email: string) {
        const staff = await this.prisma.staff.findFirst({
            where: {
                email,
                deletedAt: null,
            },
            select: {
                id: true,
                email: true,
                fullName: true,
                password: true,
                tokens: {
                    select: {
                        token: true,
                        expiredAt: true,
                        usedAt: true,
                    },
                    where: {
                        type: 'RESET_PASSWORD',
                    },
                    orderBy: {
                        expiredAt: 'desc',
                    },
                },
            },
        });

        // If staff does not exist
        if (!staff) {
            throw new HttpException('api-messages:staff-not-found', HttpStatus.NOT_FOUND);
        }
        // If password is not set (not verified yet)
        if (!staff.password) {
            throw new CustomException('api-messages:account-not-verified', HttpStatus.CONFLICT, {
                unverified: true,
            });
        }

        // Use transactions to make sure that all queries are executed
        return this.prisma.$transaction(async (tx) => {
            // 1. Invalidate all existing reset tokens
            await tx.staffToken.updateMany({
                where: {
                    staffId: staff.id,
                    type: 'RESET_PASSWORD',
                },
                data: {
                    isValid: false,
                },
            });

            // 2. Create new reset token
            // Reset token will be expired after 1 hour using dayjs
            const resetToken = uuidv4();
            const resetTokenExpiredAt = dayjs().add(1, 'hour');

            // 3. Send email to staff
            // Send verification email
            const emailResponse = await this.emailService.staffResetPassword(staff.email, {
                name: staff.fullName,
                staffId: staff.id,
                resetToken: resetToken,
            });

            // If email failed to send, throw error
            if (!emailResponse.success) {
                throw new HttpException('api-messages:email-failed-to-send', HttpStatus.INTERNAL_SERVER_ERROR);
            }

            // 4. Update staff reset token
            const updatedStaff = await tx.staff.update({
                where: {
                    id: staff.id,
                },
                select: {
                    id: true,
                    email: true,
                },
                data: {
                    tokens: {
                        create: {
                            token: resetToken,
                            type: 'RESET_PASSWORD',
                            expiredAt: resetTokenExpiredAt.toDate(),
                        },
                    },
                },
            });

            return updatedStaff;
        });
    }

    async verifyResetToken(staffId: string, token: string) {
        const staff = await this.prisma.staff.findFirst({
            where: {
                id: staffId,
                deletedAt: null,
            },
            select: {
                id: true,
                email: true,
                tokens: {
                    select: {
                        token: true,
                        expiredAt: true,
                        usedAt: true,
                    },
                    where: {
                        type: 'RESET_PASSWORD',
                        token,
                        isValid: true,
                    },
                },
            },
        });

        if (!staff) {
            throw new HttpException('api-messages:staff-not-found', HttpStatus.NOT_FOUND);
        }

        if (staff.tokens.length === 0 || dayjs().isAfter(dayjs(staff.tokens[0].expiredAt))) {
            throw new HttpException('api-messages:invalid-token', HttpStatus.UNAUTHORIZED);
        }

        delete staff.tokens;

        return staff;
    }

    async resetPassword(staffId: string, token: string, password: string) {
        // Check if staff exists and token is valid
        const staff = await this.verifyResetToken(staffId, token);

        // Hash password
        const hashedPassword = await argon.hash(password);

        // Update staff password and invalidate the token
        const updatedStaff = await this.prisma.staff.update({
            where: {
                id: staff.id,
            },
            data: {
                password: hashedPassword,
                tokens: {
                    update: {
                        where: {
                            token,
                        },
                        data: {
                            isValid: false,
                            usedAt: dayjs().toDate(),
                        },
                    },
                },
            },
            select: {
                id: true,
                email: true,
            },
        });

        return updatedStaff;
    }

    private async signToken(payload: StaffJwtPayload): Promise<string> {
        const token = await this.jwtService.signAsync(payload);
        return token;
    }
}
