import { Body, Controller, Delete, Get, HttpCode, HttpStatus, Param, Post, Put, Query, UploadedFile, UseInterceptors } from '@nestjs/common';
import { BookOrderService } from './book-order.service';
import { Authenticator, Member } from 'src/auth/auth.decorator';
import { MemberInRequest, User } from 'src/types';
import { JoiValidationPipe } from 'src/utils/validators/joi.validation.pipe';
import { bookOrderQuerySchema } from './book-order.validator';
import { BookOrderQuery, CreateBookOrderDto } from 'src/staff/book-order/book-order.dto';
import { UpdateBookOrderDto } from './book-order.dto';
import { FileInterceptor } from '@nestjs/platform-express';

@Controller()
export class BookOrderController {
    constructor(private bookOrderService: BookOrderService) {}

    @Authenticator(User.MEMBER)
    @Get('')
    @HttpCode(HttpStatus.OK)
    async getBookListByPagination(
        @Member() member: MemberInRequest,
        @Query(new JoiValidationPipe(bookOrderQuerySchema))
        query: BookOrderQuery,
    ) {
        const bookList = await this.bookOrderService.getBookOrderByPagination(member.id, query);

        return {
            total: bookList.count,
            rows: bookList.rows,
            page: bookList.page,
        };
    }

    @Authenticator(User.MEMBER)
    @Get(':bookOrderId')
    @HttpCode(HttpStatus.OK)
    async getBookOrderById(@Member() member: MemberInRequest, @Param('bookOrderId') bookOrderId: string) {
        return await this.bookOrderService.getBookOrderById(member.id, bookOrderId);
    }

    @Authenticator(User.MEMBER)
    @Post('')
    @HttpCode(HttpStatus.CREATED)
    async createBookOrder(@Member() member: MemberInRequest, @Body() body: CreateBookOrderDto) {
        const response = await this.bookOrderService.orderBook(member.id, body);
        return response;
    }

    @Authenticator(User.MEMBER)
    @Put(':bookOrderId')
    @HttpCode(HttpStatus.OK)
    async updateBookOrder(@Member() member: MemberInRequest, @Param('bookOrderId') bookOrderId: string, @Body() body: UpdateBookOrderDto) {
        const response = await this.bookOrderService.updateBookOrder(member.id, bookOrderId, body);
        return response;
    }

    @Authenticator(User.MEMBER)
    @Post('upload')
    @HttpCode(HttpStatus.OK)
    @UseInterceptors(FileInterceptor('file'))
    async uploadBookOrderMedia(@UploadedFile() file: Express.Multer.File) {
        const response = await this.bookOrderService.uploadBookOrderMedia(file);
        return response;
    }

    @Authenticator(User.MEMBER)
    @Delete(':bookOrderId')
    @HttpCode(HttpStatus.OK)
    async deleteBookOrder(@Member() member: MemberInRequest, @Param('bookOrderId') bookOrderId: string) {
        const response = await this.bookOrderService.deleteBookOrder(member.id, bookOrderId);

        return response;
    }
}
