import { ForbiddenException, HttpException, HttpStatus, Injectable } from '@nestjs/common';
import { ConfigService } from '@nestjs/config';
import { JwtService } from '@nestjs/jwt';
import { AccountStatus } from '@prisma/client';
import * as argon from 'argon2';
import * as dayjs from 'dayjs';
import { EmailService } from 'src/email/email.service';
import { PrismaService } from 'src/prisma/prisma.service';
import { MemberJwtPayload } from 'src/types';
import { CustomException } from 'src/utils/CustomException';
import { v4 as uuidv4 } from 'uuid';

@Injectable()
export class AuthService {
    constructor(
        private prisma: PrismaService,
        private jwtService: JwtService,
        private config: ConfigService,
        private emailService: EmailService,
    ) {}

    async authenticate(token: string) {
        const decoded: MemberJwtPayload = this.jwtService.verify(token, {
            secret: this.config.get('TOKEN_SECRET'),
        });

        // Get the member details
        const member = await this.prisma.member.findUnique({
            where: {
                id: decoded.id,
            },
            select: {
                id: true,
                email: true,
                fullName: true,
                status: true,
                bookTokens: true,
            },
        });

        // If member does not exist or is deleted
        if (!member) {
            throw new HttpException('api-messages:member-not-found', HttpStatus.NOT_FOUND);
        }

        // Update last active
        await this.prisma.member.update({
            where: {
                id: decoded.id,
            },
            data: {
                lastActiveAt: dayjs().toDate(),
            },
        });

        // If the member is not active
        if (member.status !== 'ACTIVE') {
            throw new HttpException('api-messages:member-disabled', HttpStatus.UNAUTHORIZED);
        }

        return member;
    }

    async login(loginBody: LoginDto) {
        const member = await this.prisma.member.findFirst({
            where: {
                email: {
                    equals: loginBody.email,
                    mode: 'insensitive',
                },
                deletedAt: null,
            },
            select: {
                id: true,
                email: true,
                password: true,
                fullName: true,
            },
        });

        // Check if member exists
        if (!member) {
            throw new ForbiddenException('api-messages:incorrect-member-email-or-password');
        }

        // Check if member is verified
        if (!member.password) {
            throw new CustomException('api-messages:account-not-verified', HttpStatus.CONFLICT, {
                unverified: true,
            });
        }

        // Check if password is correct
        const isMatch = await argon.verify(member.password, loginBody.password);

        if (!isMatch) {
            throw new ForbiddenException('api-messages:incorrect-member-email-or-password');
        }

        const payload = {
            id: member.id,
            email: member.email,
            fullName: member.fullName,
        };

        const memberToken = await this.signToken(payload);

        return memberToken;
    }

    async tokenVerifier(memberId: string, token: string) {
        const member = await this.prisma.member.findFirst({
            where: {
                id: memberId,
                deletedAt: null,
            },
            select: {
                id: true,
                email: true,
                fullName: true,
                tokens: {
                    select: {
                        token: true,
                        type: true,
                        expiredAt: true,
                    },
                    where: {
                        type: 'CONFIRMATION',
                        token,
                        usedAt: null,
                    },
                },
            },
        });

        if (!member) {
            throw new HttpException('api-messages:member-not-found', HttpStatus.NOT_FOUND);
        }

        if (member.tokens.length === 0) {
            throw new HttpException('api-messages:invalid-token', HttpStatus.NOT_FOUND);
        }

        if (dayjs().isAfter(dayjs(member.tokens[0].expiredAt))) {
            throw new CustomException('api-messages:token-expired', HttpStatus.BAD_REQUEST, {
                verificationExpired: true,
                email: member.email,
            });
        }

        return member;
    }

    async verifyMember(body: VerifyMemberDto) {
        const { id, password, token } = body;

        const member = await this.prisma.member.findFirst({
            where: {
                id,
                deletedAt: null,
            },
            select: {
                id: true,
                fullName: true,
                email: true,
                tokens: {
                    select: {
                        token: true,
                        expiredAt: true,
                    },
                    where: {
                        type: 'CONFIRMATION',
                        token,
                        usedAt: null,
                    },
                },
            },
        });

        if (!member) {
            throw new HttpException('api-messages:member-not-found', HttpStatus.NOT_FOUND);
        }

        if (member.tokens.length === 0) {
            throw new HttpException('api-messages:invalid-token', HttpStatus.NOT_FOUND);
        }

        if (dayjs().isAfter(dayjs(member.tokens[0].expiredAt))) {
            throw new CustomException('api-messages:token-expired', HttpStatus.BAD_REQUEST, {
                verificationExpired: true,
                email: member.email,
            });
        }

        // Hash password
        const hashedPassword = await argon.hash(password);

        // Update member
        const updatedMember = await this.prisma.member.update({
            where: {
                id: member.id,
            },
            data: {
                status: AccountStatus.ACTIVE,
                password: hashedPassword,
                tokens: {
                    update: {
                        where: {
                            token,
                        },
                        data: {
                            usedAt: dayjs().toDate(),
                        },
                    },
                },
            },
            select: this.prisma.createSelect(['id', 'fullName', 'preferredName', 'address', 'email', 'dateOfBirth', 'phoneNumber']),
        });

        return updatedMember;
    }

    async forgotPassword(email: string) {
        const member = await this.prisma.member.findFirst({
            where: {
                email,
                deletedAt: null,
            },
            select: {
                id: true,
                email: true,
                fullName: true,
                password: true,
                tokens: {
                    select: {
                        token: true,
                        expiredAt: true,
                        usedAt: true,
                    },
                    where: {
                        type: 'CONFIRMATION',
                    },
                    orderBy: {
                        expiredAt: 'desc',
                    },
                },
            },
        });

        // If member does not exist
        if (!member) {
            throw new HttpException('api-messages:member-not-found', HttpStatus.NOT_FOUND);
        }
        // If password is not set (not verified yet)
        if (!member.password) {
            throw new CustomException('api-messages:account-not-verified', HttpStatus.CONFLICT, {
                unverified: true,
            });
        }

        // Use transactions to make sure that all queries are executed
        return this.prisma.$transaction(async (tx) => {
            // 1. Invalidate all existing reset tokens
            await tx.memberToken.updateMany({
                where: {
                    memberId: member.id,
                    type: 'RESET_PASSWORD',
                },
                data: {
                    isValid: false,
                },
            });

            // 2. Create new reset token
            // Reset token will be expired after 1 hour using dayjs
            const resetToken = uuidv4();
            const resetTokenExpiredAt = dayjs().add(1, 'hour');

            // 3. Send email to member
            // Send verification email
            const emailResponse = await this.emailService.memberResetPassword(member.email, {
                name: member.fullName,
                memberId: member.id,
                resetToken: resetToken,
            });

            // If email failed to send, throw error
            if (!emailResponse.success) {
                throw new HttpException('api-messages:email-failed-to-send', HttpStatus.INTERNAL_SERVER_ERROR);
            }

            // 4. Update member reset token
            const updatedMember = await tx.member.update({
                where: {
                    id: member.id,
                },
                select: {
                    id: true,
                    email: true,
                },
                data: {
                    tokens: {
                        create: {
                            token: resetToken,
                            type: 'RESET_PASSWORD',
                            expiredAt: resetTokenExpiredAt.toDate(),
                        },
                    },
                },
            });

            return updatedMember;
        });
    }

    async verifyResetToken(memberId: string, token: string) {
        const member = await this.prisma.member.findFirst({
            where: {
                id: memberId,
                deletedAt: null,
            },
            select: {
                id: true,
                email: true,
                tokens: {
                    select: {
                        token: true,
                        expiredAt: true,
                        usedAt: true,
                    },
                    where: {
                        type: 'RESET_PASSWORD',
                        token,
                        isValid: true,
                    },
                },
            },
        });

        if (!member) {
            throw new HttpException('api-messages:member-not-found', HttpStatus.NOT_FOUND);
        }

        if (member.tokens.length === 0 || dayjs().isAfter(dayjs(member.tokens[0].expiredAt))) {
            throw new HttpException('api-messages:invalid-token', HttpStatus.UNAUTHORIZED);
        }

        delete member.tokens;

        return member;
    }

    async resetPassword(memberId: string, token: string, password: string) {
        // Check if member exists and token is valid
        const member = await this.verifyResetToken(memberId, token);

        // Hash password
        const hashedPassword = await argon.hash(password);

        // Update member password and invalidate the token
        const updatedMember = await this.prisma.member.update({
            where: {
                id: member.id,
            },
            data: {
                password: hashedPassword,
                tokens: {
                    update: {
                        where: {
                            token,
                        },
                        data: {
                            isValid: false,
                            usedAt: dayjs().toDate(),
                        },
                    },
                },
            },
            select: {
                id: true,
                email: true,
            },
        });

        return updatedMember;
    }

    private async signToken(payload: MemberJwtPayload): Promise<string> {
        const token = await this.jwtService.signAsync(payload);
        return token;
    }
}
