import { Body, Controller, Get, HttpCode, HttpStatus, Param, Post, Res } from '@nestjs/common';
import { Response } from 'express';
import { COOKIE_OPTIONS, USER_TOKEN_COOKIE_NAME } from 'src/config';
import { JoiValidationPipe } from 'src/utils/validators/joi.validation.pipe';
import { MemberSharedService } from '../member-shared.service';
import { memberEmailSchema, createMemberSchema } from '../member-shared.validator';
import { AuthService } from './auth.service';
import { authenticationSchema, loginSchema } from './auth.validator';

@Controller()
export class AuthController {
    constructor(
        private authService: AuthService,
        private sharedService: MemberSharedService,
    ) {}

    @Post('email-check')
    async emailCheck(@Body(new JoiValidationPipe(memberEmailSchema)) body: MemberEmailDto) {
        const response = await this.sharedService.emailCheck(body.email);
        return response;
    }

    @Post('signup')
    async signUp(@Body(new JoiValidationPipe(createMemberSchema)) body: CreateMemberDto) {
        const createdMember = await this.sharedService.createMember(body);
        return createdMember;
    }

    @Post('authenticate')
    async authenticate(@Body(new JoiValidationPipe(authenticationSchema)) body: MemberAuthenticateDto) {
        const member = await this.authService.authenticate(body.token);
        return member;
    }

    @Post('login')
    async login(@Body(new JoiValidationPipe(loginSchema)) body: LoginDto, @Res() res: Response) {
        const token = await this.authService.login(body);
        res.cookie(USER_TOKEN_COOKIE_NAME.member, token, COOKIE_OPTIONS).status(200).json({
            success: true,
            message: 'api-messages:login-success',
        });
    }

    @Post('logout')
    async logout(@Res() res: Response) {
        res.clearCookie(USER_TOKEN_COOKIE_NAME.member, { domain: COOKIE_OPTIONS.domain });
        res.status(200).json({
            success: true,
            message: 'api-messages:logout-success',
        });
    }

    @Get('tokenVerifier/:memberId/:token')
    async tokenVerifier(@Param('memberId') memberId: string, @Param('token') token: string) {
        const member = await this.authService.tokenVerifier(memberId, token);
        return member;
    }

    @Post('verify/:memberId/:token')
    async verifyMember(@Param('memberId') memberId: string, @Param('token') token: string, @Body() body: { password: string }) {
        const { password } = body;
        const verifiedMember = await this.authService.verifyMember({ id: memberId, password, token });
        return verifiedMember;
    }

    @Post('forgot-password')
    async forgotPassword(@Body(new JoiValidationPipe(memberEmailSchema)) body: MemberEmailDto) {
        const { email } = body;
        return await this.authService.forgotPassword(email);
    }

    @Get('verify-reset-token/:memberId/:token')
    async verifyResetToken(@Param('memberId') memberId: string, @Param('token') token: string) {
        const response = await this.authService.verifyResetToken(memberId, token);
        return response;
    }

    @Post('reset-password/:memberId/:token')
    async resetPassword(@Param('memberId') memberId: string, @Param('token') token: string, @Body('password') password: string) {
        const response = await this.authService.resetPassword(memberId, token, password);
        return response;
    }

    @Post('resend-verification-email')
    @HttpCode(HttpStatus.OK)
    async resendVerificationEmail(@Body(new JoiValidationPipe(memberEmailSchema)) body: MemberEmailDto) {
        await this.sharedService.resendVerificationEmail(body.email);
        return {
            success: true,
            message: 'api-messages:verification-email-sent',
        };
    }
}
