import { HttpException, HttpStatus, Injectable } from '@nestjs/common';
import { Prisma } from '@prisma/client';
import { PrismaService } from 'src/prisma/prisma.service';

@Injectable()
export class ActivityLogService {
    constructor(private prisma: PrismaService) {}
    async writeLogger({ targetId, executorName, staffId, description, reason, action, data, memberId }: ActivityLogger) {
        const affectedTarget = `${targetId.tableName}:${targetId.id}`;

        const writeToLogger = await this.prisma.activityLog.create({
            data: {
                targetId: affectedTarget,
                executorName,
                staffId,
                memberId,
                description,
                action,
                reason: reason || '',
                data: data || Prisma.JsonNull,
            },
        });

        if (!writeToLogger) {
            throw new HttpException('api-messages:failed-to-create-activity-log', HttpStatus.BAD_REQUEST);
        }

        return writeToLogger;
    }

    async getActivityLog(targetId: string, page: number, cursor: string | null) {
        const activityLog = await this.prisma.$transaction([
            this.prisma.activityLog.count({
                where: {
                    targetId: {
                        startsWith: targetId,
                    },
                },
            }),

            this.prisma.activityLog.findMany({
                select: {
                    id: true,
                    executorName: true,
                    action: true,
                    description: true,
                    reason: true,
                    createdAt: true,
                },
                where: {
                    targetId: {
                        startsWith: targetId,
                    },
                    createdAt: cursor ? { lt: new Date(cursor) } : undefined,
                },
                take: 10,
                orderBy: {
                    createdAt: 'desc',
                },
            }),
        ]);

        const nextCursor = activityLog[1].length > 0 ? activityLog[1][activityLog[1].length - 1].createdAt.toISOString() : null;

        return {
            nextCursor,
            count: activityLog[0],
            rows: activityLog[1],
        };
    }
}
