import usePagination from '@/hooks/usePagination';
import { deleteStudyRequest, getAllStudyRequestList, rejectStudyRequest } from '@/services/book-request';
import { AxiosErrorResponse, BasePageProps } from '@/types';
import { authentication, conditionalReturn, errorMessageFormatter, localeLinkGenerator } from '@/utils';
import { useMutation, useQuery, useQueryClient } from '@tanstack/react-query';
import { Button, Form, Table, Tag, Tooltip } from 'antd';
import { useTranslation } from 'next-i18next';
import { toast } from 'react-toastify';
import FilterDrawer from '../../components/book/modals/StudyRequestFilter';
import { ColumnsType } from 'antd/es/table';
import { StudyRequest } from '@/types/book-request';
import { useState } from 'react';
import ConfirmationModal from '@/components/modals/ConfirmationModal';
import ApproveRequestModal from '../../components/book/modals/ApproveRequest';
import Layout from '@/components/layouts';
import { GetServerSideProps, NextPage } from 'next';
import { serverSideTranslations } from 'next-i18next/serverSideTranslations';
import ColumnSelector from '@/components/modals/ColumnSelector';
import Link from 'next/link';
import { dateTimeTransformer } from '@/utils/timeTransformer';
import { AiOutlineExclamationCircle } from 'react-icons/ai';

const StudyRequestPage: NextPage<BasePageProps> = ({ staff }) => {
    const { t } = useTranslation(['book-request', 'common', 'messages', 'layout']);
    const [pagination, setPagination, paginationOnChange] = usePagination();
    const [filterStudyRequestForm] = Form.useForm();
    const [memberId, setMemberId] = useState<string>('');
    const [bookId, setBookId] = useState<string>('');
    const [requestId, setRequestId] = useState<string>('');
    const [isApproveRequestModalOpen, setIsApproveRequestModalOpen] = useState<boolean>(false);
    const [selectedColumn, setSelectedColumn] = useState<string[]>(['bookName', 'createdAt']);
    const queryClient = useQueryClient();

    // Query
    const bookRequestListQuery = useQuery({
        queryKey: ['study-book-request', 'pagination', pagination],
        queryFn: async () => {
            const filterValues = filterStudyRequestForm.getFieldsValue();

            const query = {
                ...filterValues,
                page: pagination.page,
                pageSize: pagination.pageSize,
                sortField: pagination.sortField,
                sortOrder: pagination.sortOrder,
            };

            const response = await getAllStudyRequestList(query);

            setPagination((prev) => {
                return {
                    ...prev,
                    page: response.data?.page,
                    total: response.data?.total,
                };
            });

            return response.data?.rows;
        },
        onError: (error: AxiosErrorResponse & Error) => {
            toast.error(t(errorMessageFormatter(error)));
        },
    });

    // Mutation
    const rejectStudyRequestMutation = useMutation({
        mutationFn: async (reason: string) => {
            const response = await rejectStudyRequest(memberId, requestId, reason);
            return response;
        },
        onSuccess() {
            queryClient.invalidateQueries(['study-request']);
            bookRequestListQuery.refetch();
        },
    });

    const deleteStudyRequestMutation = useMutation({
        mutationFn: async () => {
            const response = await deleteStudyRequest(memberId, requestId);
            return response;
        },
        onSuccess() {
            queryClient.invalidateQueries(['study-request']);
            bookRequestListQuery.refetch();
        },
    });

    // Functions
    const onResetFilterHandler = () => {
        filterStudyRequestForm.resetFields();
        bookRequestListQuery.refetch();
    };

    const openApproveRequestModal = () => {
        setIsApproveRequestModalOpen(true);
    };

    const rejectStudyRequestHandler = (reason: string) => {
        toast.promise(rejectStudyRequestMutation.mutateAsync(reason), {
            pending: t('messages:rejecting-book-request'),
            success: t('messages:book-request-rejected'),
            error: {
                render({ data }) {
                    return t(errorMessageFormatter(data as AxiosErrorResponse));
                },
            },
        });
    };

    const deleteStudyRequestHandler = () => {
        toast.promise(deleteStudyRequestMutation.mutateAsync(), {
            pending: t('messages:deleting-book-request'),
            success: t('messages:book-request-deleted'),
            error: {
                render({ data }) {
                    return t(errorMessageFormatter(data as AxiosErrorResponse));
                },
            },
        });
    };

    // Data
    const columns: ColumnsType<StudyRequest> = [
        {
            title: t('member-name'),
            dataIndex: 'fullName',
            key: 'fullName',
            render: (_: unknown, bookRequest: StudyRequest) => (
                <Link className="font-bold" href={`member/${bookRequest.member.id}`}>
                    {bookRequest.member?.fullName}
                </Link>
            ),
        },
        ...conditionalReturn(selectedColumn.includes('bookName'), [
            {
                title: t('book-name'),
                dataIndex: 'bookName',
                key: 'bookName',
                width: 200,
                render: (_: unknown, bookRequest: StudyRequest) => {
                    return <Link href={`book/${bookRequest.book.id}`}>{bookRequest.book?.name}</Link>;
                },
            },
        ]),
        ...conditionalReturn(selectedColumn.includes('email'), [
            {
                title: t('email'),
                dataIndex: 'email',
                key: 'email',
                render: (_: unknown, bookRequest: StudyRequest) => {
                    return bookRequest.member?.email;
                },
            },
        ]),
        ...conditionalReturn(selectedColumn.includes('phoneNumber'), [
            {
                title: t('phone-number'),
                dataIndex: 'phoneNumber',
                key: 'phoneNumber',
                render: (_: unknown, bookRequest: StudyRequest) => {
                    return bookRequest.member?.phoneNumber;
                },
            },
        ]),
        ...conditionalReturn(selectedColumn.includes('createdAt'), [
            {
                title: t('common:created-at'),
                dataIndex: 'createdAt',
                key: 'createdAt',
                width: 180,
                render: (_: unknown, bookRequest: StudyRequest) => {
                    return dateTimeTransformer(bookRequest.createdAt);
                },
            },
        ]),
        {
            title: t('status'),
            dataIndex: 'status',
            key: 'status',
            width: 130,
            render: (_: unknown, bookRequest) => {
                let tagColor, tagText;

                switch (bookRequest?.status) {
                    case 'PENDING':
                        tagColor = 'yellow';
                        tagText = t('pending');
                        break;
                    case 'JOINED':
                        tagColor = 'green';
                        tagText = t('joined');
                        break;
                    case 'CANCELLED':
                        tagColor = 'red';
                        tagText = t('cancelled');
                        break;
                    default:
                        tagColor = 'red';
                        tagText = t('rejected');
                }

                return (
                    <div className="flex items-center">
                        <Tag color={tagColor}>{tagText}</Tag>
                        {bookRequest?.status === 'REJECTED' && (
                            <Tooltip
                                title={
                                    <div className="flex flex-col gap-1 p-2">
                                        <span className="font-bold">{t('reason-for-rejection')}</span>
                                        <span>{bookRequest?.reason}</span>
                                    </div>
                                }
                            >
                                <AiOutlineExclamationCircle className="text-sm md:text-md" />
                            </Tooltip>
                        )}
                    </div>
                );
            },
        },
        {
            title: t('action'),
            dataIndex: 'id',
            key: 'id',
            width: 205,
            render: (id: string, bookRequest) => {
                return bookRequest?.status === 'PENDING' && staff.role.STUDY_REQUEST_UPDATE ? (
                    <div className="flex">
                        <Button
                            type="primary"
                            className="mr-2"
                            onClick={() => {
                                setBookId(bookRequest.book.id);
                                setMemberId(bookRequest.member.id);
                                openApproveRequestModal();
                            }}
                        >
                            {t('approve')}
                        </Button>
                        <ConfirmationModal
                            title={t('common:reject-confirmation')}
                            message={t('are-you-sure-you-want-to-reject-this-book-request?')}
                            okText={t('common:reject')}
                            onOk={(reason) => {
                                setRequestId(id);
                                setMemberId(bookRequest.member.id);
                                rejectStudyRequestHandler(reason as string);
                            }}
                            reason={true}
                            okButtonProps={{
                                danger: true,
                            }}
                        >
                            <Button loading={rejectStudyRequestMutation.isLoading} danger>
                                {t('reject')}
                            </Button>
                        </ConfirmationModal>
                        <ConfirmationModal
                            title={t('common:delete-confirmation')}
                            message={t('are-you-sure-you-want-to-delete-this-book-request?')}
                            okText={t('common:delete')}
                            onOk={() => {
                                setRequestId(id);
                                setMemberId(bookRequest.member.id);
                                deleteStudyRequestHandler();
                            }}
                            okButtonProps={{
                                danger: true,
                            }}
                        >
                            <Button loading={deleteStudyRequestMutation.isLoading} type="primary" danger className="ml-2">
                                {t('delete')}
                            </Button>
                        </ConfirmationModal>
                    </div>
                ) : (
                    <ConfirmationModal
                        title={t('common:delete-confirmation')}
                        message={t('are-you-sure-you-want-to-delete-this-book-request?')}
                        okText={t('common:delete')}
                        onOk={() => {
                            setRequestId(id);
                            setMemberId(bookRequest.member.id);
                            deleteStudyRequestHandler();
                        }}
                        okButtonProps={{
                            danger: true,
                        }}
                    >
                        <Button loading={deleteStudyRequestMutation.isLoading} type="primary" danger>
                            {t('delete')}
                        </Button>
                    </ConfirmationModal>
                );
            },
        },
    ];

    const columnOptions = [
        {
            label: t('book-name'),
            value: 'bookName',
        },
        {
            label: t('email'),
            value: 'email',
        },
        {
            label: t('phone-number'),
            value: 'phoneNumber',
        },
        {
            label: t('common:created-at'),
            value: 'createdAt',
        },
    ];

    const breadCrumbItems = [
        {
            label: t('layout:study-group-request'),
            path: '/study-group-request',
        },
    ];

    return (
        <Layout
            staff={staff}
            breadCrumbItems={breadCrumbItems}
            seoConfig={{
                title: t('study-request'),
            }}
            withBackground
            activeMenu={['study-group-request']}
        >
            <div>
                <div className="flex mb-4">
                    <FilterDrawer
                        filterStudyRequestForm={filterStudyRequestForm}
                        onSearch={() => bookRequestListQuery.refetch()}
                        onReset={onResetFilterHandler}
                        loading={bookRequestListQuery.isLoading}
                        filterByBookName={true}
                    />

                    <Button type="link" className="list-none" onClick={onResetFilterHandler}>
                        {t('common:reset-filter')}
                    </Button>
                </div>
                <div className="table_container">
                    {/* Column Selector */}
                    <div className="flex justify-end config_container mb-4">
                        <ColumnSelector options={columnOptions} column={selectedColumn} setColumn={setSelectedColumn} />
                    </div>
                    <Table
                        columns={columns}
                        dataSource={bookRequestListQuery.data}
                        loading={bookRequestListQuery.isLoading}
                        rowKey={(record) => record.id}
                        scroll={{ x: 1000 }}
                        onChange={paginationOnChange}
                        pagination={{
                            current: pagination.page,
                            pageSize: pagination.pageSize,
                            defaultPageSize: 1,
                            showSizeChanger: true,
                            pageSizeOptions: [10, 25, 50, 100],
                            showTotal: (total, range) => t('common:pagination', { range0: range[0], range1: range[1], total: total }),
                            total: pagination.total,
                        }}
                    />
                </div>
                <ApproveRequestModal
                    open={isApproveRequestModalOpen}
                    setOpen={setIsApproveRequestModalOpen}
                    refetch={() => bookRequestListQuery.refetch()}
                    memberId={memberId}
                    bookId={bookId}
                />
            </div>
        </Layout>
    );
};

export default StudyRequestPage;

export const getServerSideProps: GetServerSideProps = async ({ locale, req, resolvedUrl }) => {
    try {
        const authResponse = await authentication(req, 'STUDY_REQUEST_VIEW');

        return {
            props: {
                staff: authResponse,
                ...(await serverSideTranslations(locale as string, ['book-request', 'layout', 'common', 'messages', 'auth', 'api-messages'])),
            },
        };
    } catch (error: any) {
        if (error.response?.data?.unauthorized) {
            return {
                redirect: {
                    destination: localeLinkGenerator(locale, `/unauthorized`),
                    permanent: false,
                },
            };
        }

        return {
            redirect: {
                destination: localeLinkGenerator(locale, `/?redirect=/${resolvedUrl}`),
                permanent: false,
            },
        };
    }
};
