import { useState } from 'react';
import { toast } from 'react-toastify';
import { GetServerSideProps, NextPage } from 'next';
import { Trans, useTranslation } from 'next-i18next';
import { serverSideTranslations } from 'next-i18next/serverSideTranslations';
import Link from 'next/link';
import { PlusCircleOutlined } from '@ant-design/icons';
import { useMutation, useQuery } from '@tanstack/react-query';
import { Button, Form, MenuProps, Space, Table, TableColumnProps } from 'antd';
import dayjs from 'dayjs';
import MemberStatusTag from '@/components/Status';
import AddMemberModal from '@/components/member/modals/AddMember';
import FilterDrawer from '@/components/member/modals/Filter';
import ColumnSelector from '@/components/modals/ColumnSelector';
import ConfirmationModal from '@/components/modals/ConfirmationModal';
import ActionDropdown from '@/components/ui/ActionDropdown';
import usePagination from '@/hooks/usePagination';
import Layout from '@/components/layouts';
import { deleteMember, getMemberListByPagination, resendVerificationEmail, updateMemberStatus } from '@/services/member';
import { Member, MemberStatus, AxiosErrorResponse, BasePageProps, TokenType } from '@/types';
import { authentication } from '@/utils/authentication';
import conditionalReturn from '@/utils/conditionalReturn';
import { errorMessageFormatter } from '@/utils/errorFormatter';
import localeLinkGenerator from '@/utils/localeLinkGenerator';
import ExportData from '@/components/member/ExportData';
import { dateTimeTransformer, dateTransformer } from '@/utils/timeTransformer';

const MemberPage: NextPage<BasePageProps> = ({ staff }) => {
    const { t } = useTranslation(['member', 'common', 'auth']);
    const [pagination, setPagination, paginationOnChange] = usePagination<Member>();
    const [filterMemberForm] = Form.useForm<Member>();
    const [selectedColumn, setSelectedColumn] = useState<string[]>(['fullName', 'bookTokens', 'email', 'status', 'lastActiveAt']);
    const [isAddMemberModalOpen, setIsAddMemberModalOpen] = useState(false);

    // Query
    const memberQuery = useQuery({
        queryKey: ['member', 'pagination', pagination],
        queryFn: async () => {
            // Get the form values
            let searchedValue = filterMemberForm.getFieldsValue();

            const query = {
                ...searchedValue,
                page: pagination.page,
                pageSize: pagination.pageSize,
                sortField: pagination.sortField,
                sortOrder: pagination.sortOrder,
            };

            const res = await getMemberListByPagination(query);

            setPagination((prevValue) => {
                return {
                    ...prevValue,
                    page: res.data.page,
                    total: res.data.total,
                };
            });

            return res.data.rows;
        },
        onError: (error: AxiosErrorResponse & Error) => {
            toast.error(t(errorMessageFormatter(error)));
        },
    });

    // Mutations
    const updateMemberStatusMutation = useMutation({
        mutationFn: async (memberId: string) => {
            const res = await updateMemberStatus(memberId);
            return res.data;
        },
        onSuccess: () => {
            memberQuery.refetch();
        },
    });

    const deleteMemberMutation = useMutation({
        mutationFn: async (memberId: string) => {
            const res = await deleteMember(memberId);
            return res.data;
        },
        onSuccess: () => {
            memberQuery.refetch();
        },
    });

    const resendVerificationEmailMutation = useMutation({
        mutationFn: async (email: string) => {
            const res = await resendVerificationEmail(email);
            return res.data;
        },
        onSuccess: () => {
            memberQuery.refetch();
        },
    });

    // Functions
    const onResetHandler = () => {
        filterMemberForm.resetFields();
        memberQuery.refetch();
    };

    const onUpdateMemberStatusHandler = async (memberId: string) => {
        toast.promise(updateMemberStatusMutation.mutateAsync(memberId), {
            pending: t('messages:updating-member-status'),
            success: t('messages:member-status-updated'),
            error: {
                render({ data }: any) {
                    return t(errorMessageFormatter(data));
                },
            },
        });
    };

    const onDeleteMemberHandler = async (memberId: string) => {
        toast.promise(deleteMemberMutation.mutateAsync(memberId), {
            pending: t('messages:deleting-member'),
            success: t('messages:member-deleted'),
            error: {
                render({ data }: any) {
                    return t(errorMessageFormatter(data));
                },
            },
        });
    };

    const onResendVerificationEmailHandler = (email: string) => {
        toast.promise(resendVerificationEmailMutation.mutateAsync(email), {
            pending: t('messages:resending-email-verification'),
            success: t('messages:email-verification-resend'),
            error: {
                render({ data }: any) {
                    return t(errorMessageFormatter(data));
                },
            },
        });
    };

    // Data
    const breadCrumbItems = [
        {
            label: t('member'),
            path: '/',
        },
    ];

    const seoConfig = {
        title: t('member'),
    };

    const columnOptions = [
        {
            label: t('full-name'),
            value: 'fullName',
        },
        {
            label: t('preferred-name'),
            value: 'preferredName',
        },
        {
            label: t('book-tokens'),
            value: 'bookTokens',
        },
        {
            label: t('email'),
            value: 'email',
        },
        {
            label: t('address'),
            value: 'address',
        },
        {
            label: t('date-of-birth'),
            value: 'dateOfBirth',
        },
        {
            label: t('status'),
            value: 'status',
        },
        {
            label: t('phone-number'),
            value: 'phoneNumber',
        },
        {
            label: t('last-active'),
            value: 'lastActiveAt',
        },
        {
            label: t('common:created-at'),
            value: 'createdAt',
        },
    ];

    const columns = [
        // Name
        {
            dataIndex: 'id',
            title: t('full-name'),
            sorter: true,
            render: (_: unknown, member: Member) => {
                return (
                    <Link className="font-bold" href={`/member/${member.id}`}>
                        {member.fullName}
                    </Link>
                );
            },
        },
        // Preferred Name
        ...conditionalReturn(selectedColumn.includes('preferredName'), [
            {
                dataIndex: 'preferredName',
                title: t('preferred-name'),
                sorter: true,
            },
        ]),
        ...conditionalReturn(selectedColumn.includes('bookTokens'), [
            {
                dataIndex: 'bookTokens',
                title: t('book-tokens'),
                sorter: true,
            },
        ]),
        // Email
        {
            dataIndex: 'email',
            title: t('email'),
            sorter: true,
        },
        // Address
        ...conditionalReturn(selectedColumn.includes('address'), [
            {
                dataIndex: 'address',
                title: t('address'),
                sorter: true,
            },
        ]),
        // Date of Birth
        ...conditionalReturn(selectedColumn.includes('dateOfBirth'), [
            {
                dataIndex: 'dateOfBirth',
                title: t('date-of-birth'),
                sorter: true,
                render: (dateOfBirth: string) => {
                    return dateOfBirth !== null ? dateTransformer(dateOfBirth) : '';
                },
            },
        ]),
        // Status
        {
            dataIndex: 'status',
            title: t('status'),
            sorter: true,
            render: (_: unknown, member: Member) => {
                return <MemberStatusTag user={member} />;
            },
        },
        // Phone Number
        ...conditionalReturn(selectedColumn.includes('phoneNumber'), [
            {
                title: t('phone-number'),
                dataIndex: 'phoneNumber',
                sorter: true,
            },
        ]),
        // Last Active
        ...conditionalReturn(selectedColumn.includes('lastActiveAt'), [
            {
                title: t('last-active'),
                dataIndex: 'lastActiveAt',
                width: 150,
                sorter: true,
                render: (lastActiveAt: string) => {
                    return lastActiveAt !== null ? dateTimeTransformer(lastActiveAt) : '';
                },
            },
        ]),
        // Created At
        ...conditionalReturn(selectedColumn.includes('createdAt'), [
            {
                title: t('common:created-at'),
                dataIndex: 'createdAt',
                width: 150,
                sorter: true,
                render: (createdAt: string) => {
                    return createdAt !== null ? dateTimeTransformer(createdAt) : '';
                },
            },
        ]),
        // Actions
        {
            dataIndex: 'id',
            width: 64,
            fixed: 'right',
            render: (_: unknown, member: Member) => {
                const action = member.status === MemberStatus.ACTIVE ? t('common:deactivate') : t('common:activate');
                const verificationToken = member.tokens.filter((token) => token.type === TokenType.CONFIRMATION);
                const items: MenuProps['items'] = [
                    // View Member Details
                    {
                        label: <Link href={`/member/${member.id}`}>{t('common:view')}</Link>,
                        key: 'view',
                        className: 'text-center',
                    },
                    // Resend Email Verification
                    ...conditionalReturn(
                        !member.password && (verificationToken.length === 0 || dayjs.tz(verificationToken[0]?.expiredAt).isBefore(dayjs.tz())),
                        [
                            {
                                label: (
                                    <ConfirmationModal
                                        title={t('common:resend-email-verification')}
                                        message={
                                            <Trans
                                                i18nKey={'messages:are-you-sure-you-want-to-resend-email-verification-to'}
                                                components={{ strong: <strong /> }}
                                                values={{ name: member.fullName }}
                                            />
                                        }
                                        okText={t('common:resend')}
                                        onOk={() => {
                                            onResendVerificationEmailHandler(member.email);
                                        }}
                                        reason={false}
                                        width={400}
                                    >
                                        <Button
                                            loading={resendVerificationEmailMutation.isLoading}
                                            type="text"
                                            block
                                            className="!p-0"
                                            style={{ transition: 'none' }}
                                        >
                                            {t('common:resend-email')}
                                        </Button>
                                    </ConfirmationModal>
                                ),
                                key: 'resend',
                                className: '!p-0',
                            },
                        ],
                    ),
                    // Activate / Deactivate Member
                    ...conditionalReturn(staff.role.MEMBER_UPDATE, [
                        {
                            label: (
                                <ConfirmationModal
                                    title={action}
                                    message={
                                        <Trans
                                            i18nKey={'messages:are-you-sure-you-want-to-activate-or-deactivate'}
                                            components={{ strong: <strong /> }}
                                            values={{ name: member.fullName, action: action.toLowerCase() }}
                                        />
                                    }
                                    okText={t(action)}
                                    onOk={() => {
                                        onUpdateMemberStatusHandler(member.id);
                                    }}
                                    reason={false}
                                    okButtonProps={{
                                        danger: true,
                                    }}
                                    width={400}
                                >
                                    <Button
                                        loading={updateMemberStatusMutation.isLoading}
                                        type="text"
                                        block
                                        className="!p-0"
                                        style={{ transition: 'none' }}
                                    >
                                        {t(action)}
                                    </Button>
                                </ConfirmationModal>
                            ),
                            key: 'activate',
                            className: '!p-0',
                        },
                    ]),
                    // Delete Member
                    ...conditionalReturn(staff.role.MEMBER_DELETE, [
                        {
                            label: (
                                <ConfirmationModal
                                    title={t('common:delete-confirmation')}
                                    message={
                                        <Trans
                                            i18nKey={'messages:are-you-sure-you-want-to-delete'}
                                            components={{ strong: <strong /> }}
                                            values={{ name: member.fullName }}
                                        />
                                    }
                                    okText={t('common:delete')}
                                    onOk={() => {
                                        onDeleteMemberHandler(member.id);
                                    }}
                                    reason={false}
                                    okButtonProps={{
                                        danger: true,
                                    }}
                                    width={400}
                                >
                                    <Button
                                        loading={deleteMemberMutation.isLoading}
                                        type="text"
                                        block
                                        className="!p-0 hover:!text-white"
                                        style={{ transition: 'none' }}
                                    >
                                        {t('common:delete')}
                                    </Button>
                                </ConfirmationModal>
                            ),
                            key: 'delete',
                            danger: true,
                            className: '!p-0',
                        },
                    ]),
                ];

                return (
                    <Space>
                        <ActionDropdown items={items} />
                    </Space>
                );
            },
        },
    ] as TableColumnProps<Member>[];

    return (
        <Layout
            staff={staff}
            breadCrumbItems={breadCrumbItems}
            activeMenu={['member']}
            activeDropdown={['users']}
            seoConfig={seoConfig}
            withBackground
        >
            <div className="flex justify-between mb-4 flex-col sm:flex-row">
                {/* Header */}
                <div className="flex">
                    {/* Filter Drawer */}

                    <FilterDrawer
                        filterMemberForm={filterMemberForm}
                        onReset={onResetHandler}
                        onSearch={() => {
                            memberQuery.refetch();
                        }}
                        loading={false}
                    />
                    {/* Reset Filter Button */}
                    <Button type="link" className="list-none" onClick={onResetHandler}>
                        {t('common:reset-filter')}
                    </Button>
                </div>
                {/* Add Member Button */}
                <div className="flex flex-col sm:flex-row mt-2 sm:mt-0 gap-2">
                    <ExportData />
                    {staff.role.MEMBER_CREATE && (
                        <Button type="primary" icon={<PlusCircleOutlined />} onClick={() => setIsAddMemberModalOpen(true)}>
                            {t('add-member')}
                        </Button>
                    )}
                </div>
            </div>
            <div className="table_container">
                {/* Column Selector */}
                <div className="flex justify-end config_container mb-4">
                    <ColumnSelector options={columnOptions} column={selectedColumn} setColumn={setSelectedColumn} />
                </div>
                {/* Table */}
                <Table
                    columns={columns}
                    dataSource={memberQuery.data}
                    loading={memberQuery.isLoading}
                    rowKey={(record) => record.id}
                    scroll={{ x: 1000 }}
                    onChange={paginationOnChange}
                    pagination={{
                        current: pagination.page,
                        pageSize: pagination.pageSize,
                        defaultPageSize: 1,
                        showSizeChanger: true,
                        pageSizeOptions: [10, 25, 50, 100],
                        showTotal: (total, range) => t('common:pagination', { range0: range[0], range1: range[1], total: total }),
                        total: pagination.total,
                    }}
                />
            </div>
            {/* Add Member Modal */}
            <AddMemberModal open={isAddMemberModalOpen} setOpen={setIsAddMemberModalOpen} memberQuery={memberQuery} />
        </Layout>
    );
};

export default MemberPage;

export const getServerSideProps: GetServerSideProps = async ({ locale, req, resolvedUrl }) => {
    try {
        const authResponse = await authentication(req, 'MEMBER_VIEW');

        return {
            props: {
                staff: authResponse,
                ...(await serverSideTranslations(locale as string, ['member', 'layout', 'common', 'messages', 'auth', 'api-messages'])),
            },
        };
    } catch (error: any) {
        if (error.response?.data?.unauthorized) {
            return {
                redirect: {
                    destination: localeLinkGenerator(locale, `/unauthorized`),
                    permanent: false,
                },
            };
        }

        return {
            redirect: {
                destination: localeLinkGenerator(locale, `/?redirect=/${resolvedUrl}`),
                permanent: false,
            },
        };
    }
};
