import { GetServerSideProps, NextPage } from 'next';
import { Trans, useTranslation } from 'next-i18next';
import { serverSideTranslations } from 'next-i18next/serverSideTranslations';
import Layout from '@/components/layouts';
import { AxiosErrorResponse, BasePageProps, BookTokenSubscriptionList, BookTokenSubscriptionStatus } from '@/types';
import { authentication } from '@/utils/authentication';
import localeLinkGenerator from '@/utils/localeLinkGenerator';
import { Button, Form, Skeleton, Tooltip } from 'antd';
import { ColumnsType } from 'antd/es/table';
import { MenuProps, Table } from 'antd/lib';
import ActionDropdown from '@/components/ui/ActionDropdown';
import usePagination from '@/hooks/usePagination';
import { useMutation, useQuery } from '@tanstack/react-query';
import { currencyFormatter, errorMessageFormatter } from '@/utils';
import BookTokenStatusTag from '@/components/book-token-request/BookTokenStatusTag';
import { bookTokenKeys, deleteBookTokenRequest, getBookTokenSubscriptionList } from '@/services/book-token-request';
import { useState } from 'react';
import ViewBookTokenDetails from '@/components/book-token-request/modals/ViewBookTokenDetails';
import Link from 'next/link';
import FilterDrawer from '@/components/book-token-request/modals/Filter';
import { EyeOutlined } from '@ant-design/icons';
import ExportData from '@/components/book-token-request/ExportData';
import ConfirmationModal from '@/components/modals/ConfirmationModal';
import { toast } from 'react-toastify';
import HtmlParser from '@/components/HtmlParser';
import { dateTimeTransformer, dateTransformer } from '@/utils/timeTransformer';
import { AiOutlineExclamationCircle } from 'react-icons/ai';

const Index: NextPage<BasePageProps> = ({ staff }) => {
    const { t } = useTranslation(['book-token', 'layout', 'common', 'messages']);
    const [pagination, setPagination, paginationOnChange] = usePagination<BookTokenSubscriptionList>();
    const [filterBookTokenRequestForm] = Form.useForm();

    const [selectedBookTokenId, setSelectedBookTokenId] = useState<string | null>(null);
    const [isViewBookTokenModalOpen, setIsViewBookTokenModalOpen] = useState<boolean>(false);

    const {
        isFetching: bookTokenIsFetching,
        data: bookTokenData,
        refetch: bookTokenRefetch,
    } = useQuery({
        queryKey: bookTokenKeys.pagination(pagination),
        queryFn: async () => {
            const filterValues = filterBookTokenRequestForm.getFieldsValue();

            const query = {
                ...filterValues,
                page: pagination.page,
                pageSize: pagination.pageSize,
                sortField: pagination.sortField,
                sortOrder: pagination.sortOrder,
            };

            const response = await getBookTokenSubscriptionList(query);

            setPagination((prev) => {
                return {
                    ...prev,
                    page: response.data.page,
                    total: response.data.total,
                };
            });

            return response.data.rows;
        },
    });

    const deleteBookTokenRequestMutation = useMutation({
        mutationFn: async (id: string) => {
            const response = await deleteBookTokenRequest(id);
            return response.data;
        },
        onSuccess: () => {
            bookTokenRefetch();
        },
    });

    const onDeleteHandler = (id: string) => {
        toast.promise(deleteBookTokenRequestMutation.mutateAsync(id), {
            pending: t('messages:deleting-book-token-request'),
            success: t('messages:book-token-request-deleted'),
            error: {
                render({ data }) {
                    return t(errorMessageFormatter(data as AxiosErrorResponse));
                },
            },
        });
    };

    const columns: ColumnsType<BookTokenSubscriptionList> = [
        {
            title: t('ref-no'),
            dataIndex: 'displayId',
            key: 'displayId',
            width: 160,
            render: (displayId: string, record) => {
                return (
                    <Button type="link" onClick={() => onViewBookTokenHandler(record.id)} className="p-0 font-bold">
                        {displayId}
                    </Button>
                );
            },
        },
        {
            title: t('package'),
            dataIndex: ['package', 'name'],
            key: 'package',
            width: 200,
            render: (packageName: string) => {
                return (
                    <Link href={'/site-setting'} className="font-bold">
                        {packageName}
                    </Link>
                );
            },
        },
        {
            title: t('package-description'),
            dataIndex: ['package', 'description'],
            key: 'package-description',
            width: 200,
            render: (description: string) => {
                return <HtmlParser html={description} />;
            },
        },
        {
            title: t('full-name'),
            dataIndex: 'id',
            key: 'full-name',
            width: 200,
            render: (id: string, record) => {
                return (
                    <Link href={`/member/${record.member.id}`} ref={null}>
                        <Button type="link" className="p-0 font-bold">
                            {record.member.fullName}
                        </Button>
                    </Link>
                );
            },
        },
        {
            title: t('quantity'),
            dataIndex: 'quantity',
            key: 'quantity',
            width: 130,
        },
        {
            title: t('number-of-book-tokens'),
            dataIndex: 'numberOfBookTokens',
            key: 'numberOfBookTokens',
            width: 130,
        },
        {
            title: t('common:unit-price'),
            dataIndex: ['package', 'price'],
            key: 'price',
            width: 200,
            render: (price: number) => {
                return `MYR ${currencyFormatter(price)}`;
            },
        },
        {
            title: t('common:total-amount'),
            dataIndex: ['package', 'price'],
            key: 'totalAmount',
            width: 220,
            render: (price: number, bookToken) => {
                return `MYR ${currencyFormatter(price * bookToken.quantity)}`;
            },
        },
        {
            title: t('payment-status'),
            dataIndex: 'paymentStatus',
            key: 'paymentStatus',
            width: 160,
            render: (paymentStatus: BookTokenSubscriptionStatus, bookToken) => {
                return (
                    <div className="flex items-center">
                        <BookTokenStatusTag status={paymentStatus} />
                        {paymentStatus === 'REJECTED' && (
                            <Tooltip
                                title={
                                    <div className="flex flex-col gap-1 p-2">
                                        <span className="font-bold">{t('reason-for-rejection')}</span>
                                        <span>{bookToken?.reason}</span>
                                    </div>
                                }
                            >
                                <AiOutlineExclamationCircle className="text-sm md:text-md" />
                            </Tooltip>
                        )}
                    </div>
                );
            },
        },
        {
            title: t('payment-date'),
            dataIndex: 'paymentDate',
            key: 'paymentDate',
            width: 200,
            render: (paymentDate: string) => {
                return paymentDate ? dateTransformer(paymentDate) : '-';
            },
        },
        {
            title: t('common:created-at'),
            dataIndex: 'createdAt',
            key: 'createdAt',
            width: 180,
            render: (createdAt: string) => {
                return dateTimeTransformer(createdAt);
            },
        },
        {
            title: t('common:expired-at'),
            dataIndex: 'expiredAt',
            key: 'expiredAt',
            width: 180,
            render: (expiredAt: string) => {
                return expiredAt ? dateTimeTransformer(expiredAt) : '-';
            },
        },
        {
            title: t('invoice'),
            dataIndex: 'id',
            key: 'invoice',
            width: 80,
            render: (bookTokenSubscriptionId: string) => {
                return (
                    <Link href={`/bill/invoice/${bookTokenSubscriptionId}`} target="_blank" className="flex justify-center">
                        <EyeOutlined className="text-base" />
                    </Link>
                );
            },
        },
        {
            title: t('receipt'),
            dataIndex: 'id',
            key: 'receipt',
            width: 80,
            render: (bookTokenSubscriptionId: string, bookToken) => {
                return (
                    <div className="flex justify-center">
                        {bookToken.paymentStatus === BookTokenSubscriptionStatus.PAID ? (
                            <Link href={`/bill/receipt/${bookTokenSubscriptionId}`} target="_blank" className="flex justify-center">
                                <EyeOutlined className="text-base" />
                            </Link>
                        ) : (
                            '-'
                        )}
                    </div>
                );
            },
        },
        {
            title: t('common:actions'),
            dataIndex: 'createdAt',
            key: 'action',
            width: 80,
            render: (_: unknown, record) => {
                const items: MenuProps['items'] = [
                    {
                        label: t('common:view'),
                        key: 'view',
                        onClick: () => onViewBookTokenHandler(record.id),
                        className: 'text-center',
                    },
                    {
                        label: (
                            <ConfirmationModal
                                title={t('common:delete-confirmation')}
                                message={<Trans i18nKey={'messages:are-you-sure-you-want-to-delete-this-book-token-request'} />}
                                okText={t('common:delete')}
                                onOk={() => {
                                    onDeleteHandler(record.id);
                                }}
                                reason={false}
                                okButtonProps={{
                                    danger: true,
                                }}
                                width={400}
                            >
                                <Button
                                    loading={deleteBookTokenRequestMutation.isLoading}
                                    type="text"
                                    block
                                    className="!p-0 hover:!text-white hover:!bg-transparent !w-fit !h-fit"
                                    style={{ transition: 'none' }}
                                >
                                    {t('common:delete')}
                                </Button>
                            </ConfirmationModal>
                        ),
                        key: 'delete',
                        danger: true,
                        className: 'text-center',
                    },
                ];

                return <ActionDropdown items={items} />;
            },
        },
    ];

    const onViewBookTokenHandler = (subscriptionId: string) => {
        setSelectedBookTokenId(subscriptionId);
        setIsViewBookTokenModalOpen(true);
    };

    const onResetFilterHandler = () => {
        filterBookTokenRequestForm.resetFields();
        bookTokenRefetch();
    };

    const seoConfig = {
        title: t('book-token-request'),
    };

    const breadCrumbItems = [
        {
            label: t('layout:book-token-request'),
            path: '/',
        },
    ];

    return (
        <Layout staff={staff} breadCrumbItems={breadCrumbItems} activeMenu={['book-token-request']} seoConfig={seoConfig} withBackground>
            <div className="flex flex-col sm:flex-row justify-between mb-4">
                <div className="flex mb-2 sm:mb-0">
                    <FilterDrawer
                        filterBookTokenRequestForm={filterBookTokenRequestForm}
                        onSearch={bookTokenRefetch}
                        onReset={onResetFilterHandler}
                        loading={bookTokenIsFetching}
                    />
                    <Button type="link" className="list-none" onClick={onResetFilterHandler}>
                        {t('common:reset-filter')}
                    </Button>
                </div>
                <div className="flex flex-col sm:flex-row gap-3 sm:justify-start">
                    <ExportData query={filterBookTokenRequestForm.getFieldsValue()} />
                    <Link href="/bill/quotation" target="_blank">
                        <Button type="primary" ghost>
                            {t('generate-quotation')}
                        </Button>
                    </Link>
                </div>
            </div>
            {bookTokenIsFetching ? (
                <Skeleton active />
            ) : (
                <>
                    <Table
                        columns={columns}
                        dataSource={bookTokenData}
                        loading={bookTokenIsFetching}
                        rowKey={(record) => record.id}
                        scroll={{ x: 1000 }}
                        onChange={paginationOnChange}
                        pagination={{
                            current: pagination.page,
                            pageSize: pagination.pageSize,
                            defaultPageSize: 1,
                            showSizeChanger: true,
                            pageSizeOptions: [10, 25, 50, 100],
                            showTotal: (total, range) => t('common:pagination', { range0: range[0], range1: range[1], total: total }),
                            total: pagination.total,
                        }}
                    />
                    <ViewBookTokenDetails
                        open={isViewBookTokenModalOpen}
                        setOpen={setIsViewBookTokenModalOpen}
                        selectedBookTokenId={selectedBookTokenId}
                    />
                </>
            )}
        </Layout>
    );
};

export const getServerSideProps: GetServerSideProps = async ({ req, locale, resolvedUrl }) => {
    try {
        const authResponse = await authentication(req, 'BOOK_TOKEN_VIEW');

        return {
            props: {
                staff: authResponse,
                ...(await serverSideTranslations(locale as string, ['book-token', 'layout', 'common', 'messages', 'api-messages', 'book-order'])),
            },
        };
    } catch (error: any) {
        if (error.response?.data?.unauthorized) {
            return {
                redirect: {
                    destination: localeLinkGenerator(locale, `/unauthorized`),
                    permanent: false,
                },
            };
        }

        return {
            redirect: {
                destination: localeLinkGenerator(locale, `/?redirect=/${resolvedUrl}`),
                permanent: false,
            },
        };
    }
};

export default Index;
