import { GetServerSideProps, NextPage } from 'next';
import { useTranslation } from 'next-i18next';
import { serverSideTranslations } from 'next-i18next/serverSideTranslations';
import Layout from '@/components/layouts';
import { BasePageProps } from '@/types';
import { authentication } from '@/utils/authentication';
import localeLinkGenerator from '@/utils/localeLinkGenerator';
import BirthdayModule from '@/components/birthday/BirthdayModule';
import { BirthdayProvider } from '@/providers/BirthdayContext';

const BirthdayPage: NextPage<BasePageProps> = ({ staff }) => {
    const { t } = useTranslation(['birthday', 'layout', 'common', 'messages']);

    // Data
    const breadCrumbItems = [
        {
            label: t('birthday-calendar'),
            path: '/',
        },
    ];

    const seoConfig = {
        title: t('birthday-calendar'),
    };

    return (
        <Layout staff={staff} activeMenu={['birthday-calendar']} breadCrumbItems={breadCrumbItems} seoConfig={seoConfig} withBackground>
            <BirthdayProvider>
                <BirthdayModule />
            </BirthdayProvider>
        </Layout>
    );
};

export default BirthdayPage;

export const getServerSideProps: GetServerSideProps = async ({ req, locale, resolvedUrl }) => {
    try {
        const authResponse = await authentication(req, 'BIRTHDAY_VIEW');

        return {
            props: {
                staff: authResponse,
                ...(await serverSideTranslations(locale as string, ['study-group', 'birthday', 'layout', 'common', 'messages', 'api-messages'])),
            },
        };
    } catch (error: any) {
        if (error.response?.data?.unauthorized) {
            return {
                redirect: {
                    destination: localeLinkGenerator(locale, `/unauthorized`),
                    permanent: false,
                },
            };
        }

        return {
            redirect: {
                destination: localeLinkGenerator(locale, `/?redirect=/${resolvedUrl}`),
                permanent: false,
            },
        };
    }
};
