import React from 'react';
import { AxiosErrorResponse, StudyGroupPenaltySummary } from '@/types';
import { currencyFormatter, errorMessageFormatter } from '@/utils';
import { useQuery } from '@tanstack/react-query';
import { Empty, Table } from 'antd';
import { useTranslation } from 'next-i18next';
import { toast } from 'react-toastify';
import { useRouter } from 'next/router';
import { ColumnsType } from 'antd/es/table';
import { getAllPenaltiesByStudyGroup } from '@/services/study-group';
import { useStudyGroupContext } from '@/providers/StudyGroupContext';
import ExportPenaltyData from '../ExportPenaltyData';

const PenaltyTab: React.FC = () => {
    const { t } = useTranslation(['study-group', 'layout', 'common', 'messages']);
    const { studyGroupId } = useRouter().query;
    const { studyGroup } = useStudyGroupContext();

    // Query
    const { data } = useQuery({
        queryKey: ['study-group', studyGroupId, 'penalty'],
        queryFn: async () => {
            const res = await getAllPenaltiesByStudyGroup(studyGroupId as string);

            return res.data;
        },
        onError: (error: AxiosErrorResponse & Error) => {
            toast.error(t(errorMessageFormatter(error)));
        },
    });

    // Data
    const columns: ColumnsType<StudyGroupPenaltySummary> = [
        {
            title: t('member-name'),
            dataIndex: 'memberName',
            key: 'memberName',
            render: (_: string, penalty) => {
                const fullName = penalty.member?.fullName;
                const preferredName = penalty.member?.preferredName;
                return preferredName ? `${fullName} (${preferredName})` : fullName;
            },
        },
        {
            title: t('penalty-amount'),
            dataIndex: 'amount',
            key: 'penaltyAmount',
            width: 200,
            render: (amount: number) => `MYR ${currencyFormatter(amount)}`,
        },
    ];

    const totalPenalty = data?.reduce((acc, penalty) => acc + penalty.amount, 0);

    return !data ? (
        <div className="text-center">
            <Empty description={t('no-penalty-data-available')} />
        </div>
    ) : (
        <div>
            {studyGroup.data?.status === 'COMPLETED' && (
                <div className="flex justify-end mb-2">
                    <ExportPenaltyData fileName={`Penalty - ${studyGroup.data.name}`} data={data} />
                </div>
            )}
            <Table
                dataSource={data}
                columns={columns}
                pagination={false}
                summary={() => (
                    <>
                        <Table.Summary.Row className="bg-[#fafafa]">
                            <Table.Summary.Cell index={0}>
                                <b>{t('total-penalty-amount')}</b>
                            </Table.Summary.Cell>
                            <Table.Summary.Cell index={1}>MYR {currencyFormatter(totalPenalty)}</Table.Summary.Cell>
                        </Table.Summary.Row>
                    </>
                )}
            />
        </div>
    );
};

export default PenaltyTab;
