import { Trans, useTranslation } from 'next-i18next';
import ConfirmationModal from '../modals/ConfirmationModal';
import { AxiosErrorResponse } from '@/types';
import { toast } from 'react-toastify';
import { useMutation, useQueryClient } from '@tanstack/react-query';
import { errorMessageFormatter } from '@/utils';
import { Button } from 'antd';
import { deleteStudyGroupMemberById } from '@/services/study-group';
import { useRouter } from 'next/router';

interface DeleteStudyGroupMemberActionProps {
    memberId: string;
}

const DeleteStudyGroupMemberAction: React.FC<DeleteStudyGroupMemberActionProps> = ({ memberId }) => {
    const { t } = useTranslation(['book', 'layout', 'common', 'messages']);
    const queryClient = useQueryClient();
    const router = useRouter();
    const { studyGroupId } = router.query;

    const deleteStudyGroupMemberMutation = useMutation({
        mutationFn: async () => {
            const response = await deleteStudyGroupMemberById(studyGroupId as string, memberId);
            return response.data;
        },
        onSuccess: () => {
            queryClient.invalidateQueries(['study-group', studyGroupId, 'study-group-member']);
        },
    });

    const onDeleteHandler = () => {
        toast.promise(deleteStudyGroupMemberMutation.mutateAsync(), {
            pending: t('messages:removing-study-group-member'),
            success: t('messages:study-group-member-removed'),
            error: {
                render({ data }) {
                    return t(errorMessageFormatter(data as AxiosErrorResponse));
                },
            },
        });
    };

    return (
        <ConfirmationModal
            title={t('common:remove-confirmation')}
            message={<Trans i18nKey={'messages:are-you-sure-you-want-to-remove'} components={{ strong: <strong /> }} />}
            okText={t('common:remove')}
            onOk={onDeleteHandler}
            reason={false}
            okButtonProps={{
                danger: true,
            }}
            width={400}
        >
            <Button
                loading={deleteStudyGroupMemberMutation.isLoading}
                type="link"
                ghost
                block
                className="!p-0 !h-fit hover:!text-white text-black"
                style={{ transition: 'none' }}
            >
                {t('common:remove')}
            </Button>
        </ConfirmationModal>
    );
};

export default DeleteStudyGroupMemberAction;
