import { getBooks } from '@/services/study-group';
import { AxiosErrorResponse, Book } from '@/types';
import { errorMessageFormatter } from '@/utils';
import { useQuery } from '@tanstack/react-query';
import { Col, DatePicker, Form, Input, Row, Select } from 'antd';
import dayjs from 'dayjs';
import { useTranslation } from 'next-i18next';
import { toast } from 'react-toastify';
const { RangePicker } = DatePicker;

const AddGroupForm: React.FC = () => {
    const { t } = useTranslation(['study-group', 'layout', 'common', 'messages']);

    // Query
    const getBooksQuery = useQuery({
        queryFn: async () => {
            const response = await getBooks();
            return response.data;
        },
        onError: (error: AxiosErrorResponse & Error) => {
            toast.error(t(errorMessageFormatter(error)));
        },
    });

    // Function
    const disabledDate = (current: any) => {
        return current && current < dayjs.tz().startOf('day');
    };

    return (
        <Row gutter={[16, 16]} className="mb-2">
            <Col span={24}>
                <Form.Item name="name" label={t('group-name')} rules={[{ required: true, message: t('messages:required') }]}>
                    <Input placeholder={t('enter-group-name')} size="large" />
                </Form.Item>
                <Form.Item name="bookId" label={t('book')} rules={[{ required: true, message: t('messages:required') }]}>
                    <Select size="large" placeholder={t('select-a-book')}>
                        {getBooksQuery?.data?.map((book: Book) => (
                            <Select.Option key={book.id} value={book.id}>
                                {book.name}
                            </Select.Option>
                        ))}
                    </Select>
                </Form.Item>
                <Form.Item name="groupPeriod" label={t('group-period')} rules={[{ required: true, message: t('messages:required') }]}>
                    <RangePicker size="large" className="w-full" format="DD MMM YYYY" disabledDate={disabledDate} />
                </Form.Item>
            </Col>
        </Row>
    );
};

export default AddGroupForm;
