import React from 'react';
import { AxiosErrorResponse, StudyGroupPenaltySummary } from '@/types';
import { currencyFormatter, errorMessageFormatter } from '@/utils';
import { useQuery } from '@tanstack/react-query';
import { Empty, Table } from 'antd';
import { useTranslation } from 'next-i18next';
import { toast } from 'react-toastify';
import { useStudyGroupContext } from '@/providers/StudyGroupContext';
import { ColumnsType } from 'antd/es/table';
import { getTaskMemberPenalty } from '@/services/study-group';

const PenaltyTab: React.FC = () => {
    const { t } = useTranslation(['study-group', 'layout', 'common', 'messages']);
    const { studyGroupTaskId } = useStudyGroupContext();

    // Query
    const getTaskPenaltyQuery = useQuery({
        queryKey: ['sub-task-penalty', studyGroupTaskId],
        enabled: !!studyGroupTaskId,
        queryFn: async () => {
            const res = await getTaskMemberPenalty(studyGroupTaskId);

            return res.data;
        },
        onError: (error: AxiosErrorResponse & Error) => {
            toast.error(t(errorMessageFormatter(error)));
        },
    });

    const penaltyData = getTaskPenaltyQuery.data;

    const memberPenalties = penaltyData?.penalties;

    // Data
    const columns: ColumnsType<StudyGroupPenaltySummary> = [
        {
            title: t('member-name'),
            dataIndex: 'memberName',
            key: 'memberName',
            render: (_: string, penalty) => {
                const fullName = penalty.member?.fullName;
                const preferredName = penalty.member?.preferredName;
                return preferredName ? `${fullName} (${preferredName})` : fullName;
            },
        },
        {
            title: t('penalty-amount'),
            dataIndex: 'amount',
            key: 'penaltyAmount',
            width: 200,
            render: (_: any, penalty) => `MYR ${currencyFormatter(penalty.amount)}`,
        },
    ];

    return (
        <div className="flex justify-center">
            {memberPenalties?.length ? (
                <Table
                    className="w-[80%]"
                    dataSource={memberPenalties}
                    columns={columns}
                    pagination={false}
                    summary={() => (
                        <>
                            <Table.Summary.Row className="bg-[#fafafa]">
                                <Table.Summary.Cell index={0}>
                                    <b>{t('total-penalty-amount')}</b>
                                </Table.Summary.Cell>
                                <Table.Summary.Cell index={1}>MYR {currencyFormatter(penaltyData?.totalAmount)}</Table.Summary.Cell>
                            </Table.Summary.Row>
                        </>
                    )}
                />
            ) : (
                <div className="text-center">
                    <Empty description={t('no-penalty-data-available')} />
                </div>
            )}
        </div>
    );
};

export default PenaltyTab;
