import PhoneInput from 'react-phone-input-2';
import 'react-phone-input-2/lib/style.css';
import { toast } from 'react-toastify';
import { useTranslation } from 'next-i18next';
import { useQuery } from '@tanstack/react-query';
import { Button, Col, Form, FormInstance, Input, Modal, Row, Select } from 'antd';
import { AxiosErrorResponse, BaseModalProps } from '@/types';
import { errorMessageFormatter } from '@/utils/errorFormatter';
import { getRoleData } from '@/services/data';
import { UserOutlined } from '@ant-design/icons';

interface AddStaffModalProps extends BaseModalProps {
    isLoaded: boolean;
    form: FormInstance;
    onCreate: () => void;
}

const AddStaff: React.FC<AddStaffModalProps> = ({ isLoaded, form, open, setOpen, onCreate }) => {
    const { t } = useTranslation(['staff', 'common']);

    // Role Query
    const roleQuery = useQuery({
        queryKey: ['role', 'data'],
        enabled: open,
        queryFn: async () => {
            const res = await getRoleData();

            return res.data;
        },
        onError: (error: AxiosErrorResponse & Error) => {
            toast.error(t(errorMessageFormatter(error)));
        },
        staleTime: 1000 * 60 * 5, // 5 minutes
    });

    const roleOptions =
        roleQuery.data?.map((role) => ({
            label: role.name,
            value: role.id,
        })) ?? [];

    const handleCancel = () => {
        setOpen(false);
    };

    const breakPoint = {
        xs: 24,
        sm: 24,
        md: 12,
        lg: 12,
    };

    return (
        <Modal title={t('add-staff')} open={open} onCancel={handleCancel} footer={null} width={650}>
            <Form form={form} layout="vertical" size="large" name="add_staff_form" className="mt-4">
                {/* Full Name Input */}
                <Form.Item
                    label={t('full-name')}
                    name="fullName"
                    rules={[
                        {
                            required: true,
                        },
                    ]}
                    className="mb-5"
                    labelCol={{ flex: '35px' }}
                >
                    <Input classNames={{ prefix: '!me-2' }} prefix={<UserOutlined className="site-form-item-icon" />} />
                </Form.Item>
                <Row gutter={[16, 0]}>
                    {/* Email Input */}
                    <Col {...breakPoint}>
                        <Form.Item
                            label={t('email')}
                            name="email"
                            rules={[
                                {
                                    required: true,
                                },
                                { type: 'email' },
                            ]}
                            labelCol={{ flex: '35px' }}
                        >
                            <Input classNames={{ prefix: '!me-2' }} prefix={<UserOutlined className="site-form-item-icon" />} />
                        </Form.Item>
                    </Col>
                    {/* Role Input */}
                    <Col {...breakPoint}>
                        <Form.Item
                            label={t('role')}
                            name="roleId"
                            rules={[
                                {
                                    required: true,
                                },
                            ]}
                            labelCol={{ flex: '35px' }}
                        >
                            <Select options={roleOptions} />
                        </Form.Item>
                    </Col>
                </Row>
                <Row gutter={[16, 0]}>
                    {/* Phone Number Input */}
                    <Col {...breakPoint}>
                        <Form.Item label={t('phone-number')} name="phoneNumber" rules={[{ required: true }]} labelCol={{ flex: '35px' }}>
                            <PhoneInput
                                inputProps={{
                                    required: true,
                                }}
                                inputClass="form-control"
                                country={'my'}
                                inputStyle={{ width: '100%', height: '40px', borderRadius: '8px' }}
                                buttonStyle={{ borderRadius: '8px 0px 0px 8px' }}
                            />
                        </Form.Item>
                    </Col>
                </Row>
                <div className="flex justify-between mt-3 gap-x-3">
                    <Button type="default" block onClick={handleCancel}>
                        {t('common:cancel')}
                    </Button>
                    <Button type="primary" disabled={isLoaded} block htmlType="submit" onClick={onCreate}>
                        {t('common:add')}
                    </Button>
                </div>
            </Form>
        </Modal>
    );
};

export default AddStaff;
