import { useState } from 'react';
import 'react-phone-input-2/lib/style.css';
import { toast } from 'react-toastify';
import { useTranslation } from 'next-i18next';
import { UseQueryResult, useMutation } from '@tanstack/react-query';
import { Button, Form, Input, Modal } from 'antd';
import PasswordHintContainer from '@/components/ui/input/password/PasswordHintContainer';
import { updateMemberPassword } from '@/services/member';
import { Member, BaseModalProps, UpdateMemberPasswordParams, AxiosErrorResponse } from '@/types';
import { errorMessageFormatter } from '@/utils/errorFormatter';
import { passwordRegex } from '@/utils/regex';

interface PasswordModalProps extends BaseModalProps {
    member: Member;
    memberQuery: UseQueryResult<Member | null>;
}

const PasswordModal: React.FC<PasswordModalProps> = ({ open, setOpen, member, memberQuery }) => {
    const { t } = useTranslation(['member', 'common']);
    const [form] = Form.useForm<{ password: string }>();
    const [password, setPassword] = useState<string>('');

    // Mutations
    const updateMemberPasswordMutation = useMutation({
        mutationFn: async ({ memberId, body }: { memberId: string; body: UpdateMemberPasswordParams }) => {
            const res = await updateMemberPassword(memberId, { password: body.password });
            return res.data;
        },
        onSuccess: () => {
            form.resetFields();
            setOpen(false);
            memberQuery.refetch();
        },
    });

    // Functions
    const onCancelHandler = () => {
        form.resetFields();
        setOpen(false);
    };

    const onUpdateMemberPasswordHandler = () => {
        form.validateFields().then(async (values) => {
            toast.promise(updateMemberPasswordMutation.mutateAsync({ memberId: member.id, body: values }), {
                pending: t('messages:updating-member-password'),
                success: t('messages:member-password-updated'),
                error: {
                    render({ data }) {
                        return t(errorMessageFormatter(data as AxiosErrorResponse));
                    },
                },
            });
        });
    };

    const handlePasswordOnChange = (e: React.ChangeEvent<HTMLInputElement>) => {
        e.preventDefault();
        setPassword(e.target.value);
    };

    return (
        <Modal open={open} onCancel={onCancelHandler} footer={null} title={t('edit-password')}>
            <Form layout="vertical" form={form} className="mt-4" size="large">
                <Form.Item
                    name="password"
                    label={t('password')}
                    rules={[
                        { required: true },
                        {
                            pattern: passwordRegex,
                            message: '',
                        },
                    ]}
                    className="mb-5"
                    labelCol={{ flex: '35px' }}
                >
                    <Input.Password classNames={{ prefix: '!me-2' }} onChange={handlePasswordOnChange} />
                </Form.Item>
                <div className="mt-5">
                    <PasswordHintContainer password={password} />
                </div>
                <Form.Item
                    name="confirmPassword"
                    label={t('confirm-password')}
                    dependencies={['password']}
                    rules={[
                        { required: true },
                        {
                            pattern: passwordRegex,
                            message: '',
                        },
                        {
                            validator: (_, value) => (value === password ? Promise.resolve() : Promise.reject(t('messages:password-not-match'))),
                        },
                    ]}
                    labelCol={{ flex: '35px' }}
                >
                    <Input.Password classNames={{ prefix: '!me-2' }} />
                </Form.Item>
                <div className="pt-2 flex gap-3">
                    <Button size="large" block onClick={onCancelHandler} loading={updateMemberPasswordMutation.isLoading}>
                        {t('common:cancel')}
                    </Button>
                    <Button
                        loading={updateMemberPasswordMutation.isLoading}
                        size="large"
                        type="primary"
                        block
                        onClick={onUpdateMemberPasswordHandler}
                    >
                        {t('common:update')}
                    </Button>
                </div>
            </Form>
        </Modal>
    );
};

export default PasswordModal;
