import { useEffect } from 'react';
import { toast } from 'react-toastify';
import { useTranslation } from 'next-i18next';
import { UseQueryResult, useMutation } from '@tanstack/react-query';
import { Button, Form, Input, Modal } from 'antd';
import { updateMemberEmail } from '@/services/member';
import { Member, BaseModalProps, UpdateMemberEmailParams, AxiosErrorResponse } from '@/types';
import { errorMessageFormatter } from '@/utils/errorFormatter';

interface EmailModalProps extends BaseModalProps {
    member: Member;
    memberQuery: UseQueryResult<Member | null>;
}

const EmailModal: React.FC<EmailModalProps> = ({ open, setOpen, member, memberQuery }) => {
    const { t } = useTranslation(['member', 'common']);
    const [form] = Form.useForm<{ email: string }>();
    const emailInput = Form.useWatch('email', form);

    useEffect(() => {
        form.setFieldsValue({ email: member.email });
    }, []);

    const isDiasbled = () => {
        if (!emailInput || emailInput === member.email) {
            return true;
        }

        return false;
    };

    // Mutations
    const updateMemberEmailMutation = useMutation({
        mutationFn: async ({ memberId, body }: { memberId: string; body: UpdateMemberEmailParams }) => {
            const res = await updateMemberEmail(memberId, body);
            return res.data;
        },
        onSuccess: () => {
            setOpen(false);
            memberQuery.refetch();
        },
    });

    // Functions
    const onCancelHandler = () => {
        form.resetFields();
        setOpen(false);
    };

    const onUpdateMemberEmailHandler = () => {
        form.validateFields().then(async (values) => {
            toast.promise(updateMemberEmailMutation.mutateAsync({ memberId: member.id, body: values }), {
                pending: t('messages:updating-member-email'),
                success: t('messages:member-email-updated'),
                error: {
                    render({ data }) {
                        return t(errorMessageFormatter(data as AxiosErrorResponse));
                    },
                },
            });
        });
    };

    return (
        <Modal open={open} onCancel={onCancelHandler} footer={null} title={t('edit-email')}>
            <Form layout="vertical" form={form} className="mt-4">
                <Form.Item
                    label="New Email"
                    name="email"
                    rules={[
                        { required: true },
                        {
                            type: 'email',
                        },
                    ]}
                >
                    <Input size="large" />
                </Form.Item>
                <div className="pt-2 flex gap-3">
                    <Button size="large" block onClick={onCancelHandler}>
                        {t('common:cancel')}
                    </Button>
                    <Button size="large" type="primary" disabled={isDiasbled()} block onClick={onUpdateMemberEmailHandler}>
                        {t('common:update')}
                    </Button>
                </div>
            </Form>
        </Modal>
    );
};

export default EmailModal;
