import { Button, Form, Modal } from 'antd';
import { useMutation, useQueryClient } from '@tanstack/react-query';
import { useTranslation } from 'next-i18next';
import { AxiosErrorResponse } from '@/types';
import { toast } from 'react-toastify';
import { Dispatch, SetStateAction } from 'react';
import { errorMessageFormatter } from '@/utils';
import { approveStudyRequest } from '@/services/book-request';
import ApproveRequestFormFields from '../ApproveRequestFormFields';
import React from 'react';
import { StudyGroup } from '@/types/study-group';

interface ApproveStudyRequestModalProps {
    open: boolean;
    setOpen: Dispatch<SetStateAction<boolean>>;
    refetch: () => void;
    memberId: string;
    bookId?: string;
}

const ApproveRequestModal: React.FC<ApproveStudyRequestModalProps> = ({ open, setOpen, refetch, memberId, bookId }) => {
    const { t } = useTranslation(['book-request', 'layout', 'common', 'messages']);
    const [approveStudyRequestForm] = Form.useForm<StudyGroup>();
    const [addGroup, setAddGroup] = React.useState<boolean>(false);
    const queryClient = useQueryClient();

    // Mutation
    const approveStudyRequestMutation = useMutation({
        mutationFn: async (body: StudyGroup) => {
            const response = await approveStudyRequest(bookId as string, memberId, body);
            return response;
        },
        onSuccess: () => {
            queryClient.invalidateQueries(['study-request']);
            onCancelHandler();
            refetch();
        },
    });

    // Function
    const onCancelHandler = () => {
        approveStudyRequestForm.resetFields();
        setAddGroup(false);
        setOpen(false);
    };

    const onApproveStudyRequestHandler = async () => {
        approveStudyRequestForm.validateFields().then(async (values) => {
            toast.promise(approveStudyRequestMutation.mutateAsync(values), {
                pending: t('messages:approving-request'),
                success: t('messages:request-approved'),
                error: {
                    render({ data }) {
                        return t(errorMessageFormatter(data as AxiosErrorResponse));
                    },
                },
            });
        });
        refetch();
    };

    return (
        <Modal open={open} onCancel={onCancelHandler} footer={null} title={t('approve-request')}>
            <Form form={approveStudyRequestForm} layout="vertical">
                <ApproveRequestFormFields bookId={bookId} addGroup={addGroup} setAddGroup={setAddGroup} />
                <div className="flex justify-between gap-4">
                    <Button loading={approveStudyRequestMutation.isLoading} block onClick={onCancelHandler}>
                        {t('common:cancel')}
                    </Button>
                    <Button loading={approveStudyRequestMutation.isLoading} block type="primary" onClick={onApproveStudyRequestHandler}>
                        {t('invite')}
                    </Button>
                </div>
            </Form>
        </Modal>
    );
};

export default ApproveRequestModal;
