import { SUBSCRIPTION_FILE_SIZE_LIMIT } from '@/configs';
import { bookTokenKeys, getBookTokenSubscriptionById, updateBookTokenById, uploadProofMedia } from '@/services/book-token-request';
import { mediaFormValueToResponse, mediaResponseToFormValue } from '@/transformer/media/media';
import { AxiosErrorResponse, BookTokenSubscriptionStatus, MediaFormValue, UpdateBookTokenDetails } from '@/types';
import { errorMessageFormatter, normFile } from '@/utils';
import { UploadOutlined } from '@ant-design/icons';
import { useMutation, useQuery, useQueryClient } from '@tanstack/react-query';
import { Button, DatePicker, Empty, Form, Input, InputNumber, Modal, Select, Skeleton, Upload } from 'antd';
import dayjs from 'dayjs';
import { useTranslation } from 'next-i18next';
import React from 'react';
import { Dispatch, SetStateAction } from 'react';
import { toast } from 'react-toastify';

interface ViewBookTokenDetailsProps {
    open: boolean;
    setOpen: Dispatch<SetStateAction<boolean>>;
    selectedBookTokenId: string | null;
}

const ViewBookTokenDetails: React.FC<ViewBookTokenDetailsProps> = ({ open, setOpen, selectedBookTokenId }) => {
    const { t } = useTranslation(['book-token', 'common', 'messages']);
    const [bookTokenForm] = Form.useForm<UpdateBookTokenDetails>();
    const [rejected, setRejected] = React.useState(false);
    const queryClient = useQueryClient();

    const { data, isFetching } = useQuery({
        queryKey: bookTokenKeys.id(selectedBookTokenId as string),
        queryFn: async () => {
            const response = await getBookTokenSubscriptionById(selectedBookTokenId as string);

            return response.data;
        },
        enabled: !!selectedBookTokenId && open,
        onSuccess(data) {
            const { expiredAt, paymentStatus, ...rest } = data;

            const isRejected = paymentStatus === BookTokenSubscriptionStatus.REJECTED;
            setRejected(isRejected);

            bookTokenForm.setFieldsValue({
                ...rest,
                paymentStatus: paymentStatus,
                package: data.package.name,
                expiredAt: expiredAt ? dayjs.tz(expiredAt) : undefined,
                uploadFile: data.bookTokenSubscriptionMedias.map(mediaResponseToFormValue),
            });
        },
    });

    /* Update Mutation */
    const updateBookTokenMutation = useMutation({
        mutationFn: async (body: UpdateBookTokenDetails) => {
            const response = await updateBookTokenById(selectedBookTokenId as string, body);
            return response.data;
        },
        onSuccess: () => {
            queryClient.invalidateQueries(['book-token-subscription']);
            setOpen(false);
        },
    });

    const onCancelHandler = () => {
        setOpen(false);
    };

    const onUpdateHandler = async () => {
        bookTokenForm.validateFields().then(async (values) => {
            const formattedData = values.uploadFile.map(mediaFormValueToResponse);
            const body: UpdateBookTokenDetails = {
                expiredAt: values.expiredAt,
                quantity: values.quantity,
                uploadFile: formattedData as unknown as MediaFormValue[],
                paymentStatus: values.paymentStatus,
                reason: values.reason,
                package: values.package,
            };

            toast.promise(updateBookTokenMutation.mutateAsync(body), {
                pending: t('messages:updating'),
                success: t('messages:updated'),
                error: {
                    render({ data }) {
                        return t(errorMessageFormatter(data as AxiosErrorResponse));
                    },
                },
            });
        });
    };

    return (
        <Modal footer={null} open={open} onCancel={onCancelHandler} title={data?.displayId}>
            {isFetching ? (
                <Skeleton />
            ) : !data ? (
                <Empty />
            ) : (
                <>
                    <Form form={bookTokenForm} layout="vertical">
                        <Form.Item name="package" label={t('package')}>
                            <Input size="large" disabled />
                        </Form.Item>
                        <Form.Item name="quantity" label={t('quantity')} rules={[{ required: true, message: t('messages:required') }]}>
                            <InputNumber min={1} className="w-full" size="large" />
                        </Form.Item>
                        <Form.Item name="paymentStatus" label={t('payment-status')} rules={[{ required: true, message: t('messages:required') }]}>
                            <Select
                                size="large"
                                options={[
                                    { label: t('pending'), value: BookTokenSubscriptionStatus.PENDING },
                                    { label: t('unpaid'), value: BookTokenSubscriptionStatus.UNPAID },
                                    { label: t('paid'), value: BookTokenSubscriptionStatus.PAID },
                                    { label: t('rejected'), value: BookTokenSubscriptionStatus.REJECTED },
                                ]}
                                onChange={(value) => {
                                    if (value === BookTokenSubscriptionStatus.REJECTED) {
                                        setRejected(true);
                                    } else {
                                        setRejected(false);
                                    }
                                }}
                            />
                        </Form.Item>
                        {rejected && (
                            <Form.Item name="reason" label={t('reason-for-rejection')} rules={[{ required: true, message: t('messages:required') }]}>
                                <Input.TextArea size="large" />
                            </Form.Item>
                        )}
                        <Form.Item name="expiredAt" label={t('expired-at')}>
                            <DatePicker className="w-full" size="large" value={data.expiredAt ? dayjs.tz(data.expiredAt) : undefined} />
                        </Form.Item>
                        <Form.Item
                            name="uploadFile"
                            label={t('upload-file')}
                            rules={[{ required: true, message: t('messages:required') }]}
                            getValueFromEvent={normFile}
                            valuePropName="fileList"
                        >
                            <Upload
                                multiple
                                className="!w-full"
                                maxCount={3}
                                beforeUpload={(file) => {
                                    const isLTSizeLimit = file.size / 1024 / 1024 > SUBSCRIPTION_FILE_SIZE_LIMIT;

                                    if (isLTSizeLimit) {
                                        toast.error(t('messages:file-too-large', { fileName: file.name, size: SUBSCRIPTION_FILE_SIZE_LIMIT }));
                                        return Upload.LIST_IGNORE;
                                    }

                                    return true;
                                }}
                                customRequest={(options) => uploadProofMedia('/staff/book-token/upload', options)}
                            >
                                <Button block icon={<UploadOutlined />} size="large">
                                    {t('common:upload-receipt')}
                                </Button>
                            </Upload>
                        </Form.Item>
                    </Form>
                    <div className="flex flex-col lg:flex-row gap-4">
                        <Button block loading={updateBookTokenMutation.isLoading} onClick={onCancelHandler}>
                            {t('common:cancel')}
                        </Button>
                        <Button block type="primary" loading={updateBookTokenMutation.isLoading} onClick={onUpdateHandler}>
                            {t('common:update')}
                        </Button>
                    </div>
                </>
            )}
        </Modal>
    );
};

export default ViewBookTokenDetails;
