import { HttpException, HttpStatus, Injectable } from '@nestjs/common';
import { PrismaService } from 'src/prisma/prisma.service';
import { PackageDto, PackageQuery, PenaltyDto, UpdateCompanyInformationDto } from './site-setting.dto';
import { toNumber } from 'lodash';
import { Prisma } from '@prisma/client';
import { conditionalReturn } from 'src/utils';
import { MediaService } from 'src/media/media.service';

@Injectable()
export class SiteSettingService {
    constructor(
        private prisma: PrismaService,
        private readonly media: MediaService,
    ) {}

    async getPackages(query: PackageQuery) {
        const { page, pageSize, sortField, sortOrder, name, status } = query;

        const whereOptions: Prisma.PackageWhereInput = {
            deletedAt: null,
            ...conditionalReturn(!!name, {
                name: { mode: 'insensitive', contains: name },
            }),
            ...conditionalReturn(!!status, {
                status,
            }),
        };

        const packageCount = await this.prisma.package.count({
            where: whereOptions,
        });

        const currentPage = this.prisma.pageCounter(packageCount, page, pageSize);

        const packageList = await this.prisma.package.findMany({
            where: whereOptions,
            skip: (currentPage - 1) * pageSize,
            take: pageSize,
            orderBy: {
                [!sortField ? 'createdAt' : sortField]: sortOrder ?? 'asc',
            },
        });

        return {
            count: packageCount,
            rows: packageList,
            page: currentPage,
        };
    }

    async getSinglePackage(packageId: string) {
        const response = await this.prisma.package.findFirst({
            where: {
                id: packageId,
                deletedAt: null,
            },
        });

        if (!response) {
            throw new HttpException('api-messages:package-not-found', HttpStatus.NOT_FOUND);
        }

        return response;
    }

    async addPackage(body: PackageDto) {
        const { name, price, status, numberOfBookToken, description } = body;

        const addedPackage = await this.prisma.package.create({
            data: {
                name,
                price,
                numberOfBookToken,
                description,
                status,
            },
        });

        return addedPackage;
    }

    async updatePackage(packageId: string, body: PackageDto) {
        const { name, price, status, numberOfBookToken, description } = body;

        const searchPackage = await this.prisma.package.findUnique({
            where: {
                id: packageId,
            },
        });

        if (!searchPackage) {
            throw new HttpException('api-messages:package-not-found', HttpStatus.NOT_FOUND);
        }

        const updatedPackage = await this.prisma.package.update({
            where: {
                id: packageId,
            },
            data: {
                name,
                price,
                numberOfBookToken,
                description,
                status,
            },
        });

        return updatedPackage;
    }

    async deletePackage(packageId: string) {
        const searchPackage = await this.prisma.package.findFirst({
            where: {
                id: packageId,
                deletedAt: null,
            },
        });

        if (!searchPackage) {
            throw new HttpException('api-messages:package-not-found', HttpStatus.NOT_FOUND);
        }

        const deletedPackage = await this.prisma.package.update({
            where: {
                id: searchPackage.id,
            },
            data: {
                deletedAt: new Date(),
            },
        });

        return deletedPackage;
    }

    async getPenaltyAmount() {
        const response = await this.prisma.metaData.findFirst({
            where: {
                key: 'penalty',
            },
            select: {
                value: true,
            },
        });

        if (!response) {
            throw new HttpException('api-messages:penalty-not-found', HttpStatus.NOT_FOUND);
        }

        const data = toNumber(response.value);

        return { penaltyAmount: data };
    }

    async updatePenaltyAmount(body: PenaltyDto) {
        const { penaltyAmount } = body;
        const priceToString = penaltyAmount.toString();

        const response = await this.prisma.metaData.upsert({
            where: {
                key: 'penalty',
            },
            update: {
                value: priceToString,
            },
            create: {
                key: 'penalty',
                value: priceToString,
            },
        });

        return response;
    }

    async getCompanyInformation() {
        const response = await this.prisma.metaData.findFirst({
            where: {
                key: 'company-information',
            },
            select: {
                value: true,
            },
        });

        return response.value;
    }

    async updateCompanyInformation(companyInformation: UpdateCompanyInformationDto) {
        const response = await this.prisma.metaData.upsert({
            where: {
                key: 'company-information',
            },
            update: {
                value: JSON.stringify(companyInformation),
            },
            create: {
                key: 'company-information',
                value: JSON.stringify(companyInformation),
            },
        });

        return response;
    }

    async uploadNoticeImage(file: Express.Multer.File) {
        const fileResponse = await this.media.upload(file, { isPublic: false });

        const mediaResponse = await this.prisma.media.create({
            data: fileResponse,
            select: this.prisma.createSelect(['id', 'name', 'type', 'key']),
        });

        return mediaResponse;
    }

    async getNotices() {
        const websiteContent = await this.prisma.metaData.findMany({
            where: {
                key: 'notices',
            },
        });

        return websiteContent;
    }

    async updateNotices(jsonValue: string) {
        const notices = await this.prisma.metaData.findFirst({
            where: {
                key: 'notices',
            },
        });

        let returnNotice;
        if (!notices) {
            returnNotice = await this.prisma.metaData.create({
                data: {
                    key: 'notices',
                    jsonValue,
                },
            });
        } else {
            returnNotice = await this.prisma.metaData.update({
                where: {
                    id: notices.id,
                },
                data: {
                    jsonValue,
                },
            });
        }

        return returnNotice;
    }
}
