import { HttpException, HttpStatus, Injectable } from '@nestjs/common';
import { Prisma, Role } from '@prisma/client';
import * as dayjs from 'dayjs';
import { CacheService } from 'src/cache/cache.service';
import { PrismaService } from 'src/prisma/prisma.service';
import { RolesQuery } from 'src/types';
import { CreateRoleDto, UpdateRoleDto } from './role.dto';

@Injectable()
export class RoleService {
    constructor(
        private prisma: PrismaService,
        private cache: CacheService,
    ) {}

    async getRoleList(query: RolesQuery) {
        const { page, pageSize, sortField, sortOrder, name } = query;

        const roleWhereOptions: Prisma.RoleWhereInput = {
            deletedAt: null,
            name: name ?? undefined,
        };

        const roleCount = await this.prisma.role.count({
            where: roleWhereOptions,
        });

        // If page is out of range, return the last page
        const currentPage = this.prisma.pageCounter(roleCount, page, pageSize);

        // Get filtered roles and filtered role count
        const roles = await this.prisma.role.findMany({
            take: pageSize,
            skip: (currentPage - 1) * pageSize,
            orderBy: {
                [!sortField ? 'createdAt' : sortField]: sortOrder ?? 'asc',
            },
            where: roleWhereOptions,
            select: this.prisma.createSelect<keyof Role>(['id', 'name', 'superAdmin', 'createdAt', 'updatedAt']),
        });

        return {
            total: roleCount,
            rows: roles,
            page: currentPage,
        };
    }

    async getSingleRole(roleId: string) {
        const role = await this.prisma.role.findUnique({
            where: {
                id: roleId,
                deletedAt: null,
            },
        });

        if (!role) {
            throw new HttpException('api-messages:role-not-found', HttpStatus.NOT_FOUND);
        }

        return role;
    }

    async createRole(body: CreateRoleDto) {
        const roleResponse = await this.prisma.role.create({
            data: {
                ...body,
                superAdmin: false,
            },
            select: {
                id: true,
            },
        });

        // Delete cache
        await this.cache.delete('role');

        return roleResponse;
    }

    async updateRole(roleId: string, body: UpdateRoleDto) {
        const { name, ...permissions } = body;

        const role = await this.prisma.role.findUnique({
            where: {
                id: roleId,
                deletedAt: null,
            },
        });

        if (!role) {
            throw new HttpException('api-messages:role-not-found', HttpStatus.NOT_FOUND);
        }

        // If role is superAdmin and some permissions are changed to false
        if (role.superAdmin && Object.values(permissions).includes(false)) {
            throw new HttpException('api-messages:super-admin-permissions-not-editable', HttpStatus.BAD_REQUEST);
        }

        // Delete cache
        await this.cache.delete('role');

        return await this.prisma.role.update({
            where: {
                id: roleId,
            },
            data: body,
        });
    }

    async deleteRole(roleId: string) {
        const isUserAttachedToThisRole = await this.prisma.staff.count({
            where: {
                roleId,
                deletedAt: null,
            },
        });

        if (isUserAttachedToThisRole > 0) {
            throw new HttpException('api-messages:role-in-use', HttpStatus.BAD_REQUEST);
        }

        const role = await this.prisma.role.findFirst({
            where: {
                id: roleId,
                deletedAt: null,
            },
        });

        if (!role) {
            throw new HttpException('api-messages:role-not-found', HttpStatus.NOT_FOUND);
        }

        if (role.superAdmin) {
            throw new HttpException('api-messages:role-is-not-deletable', HttpStatus.BAD_REQUEST);
        }

        await this.prisma.role.update({
            where: {
                id: roleId,
            },
            data: {
                deletedAt: dayjs().toDate(),
            },
        });

        // Delete cache
        await this.cache.delete('role');

        return role;
    }

    async restoreRole(roleId: string) {
        const role = await this.prisma.role.findUnique({
            where: {
                id: roleId,
            },
        });

        if (!role) {
            throw new HttpException('api-messages:role-not-found', HttpStatus.NOT_FOUND);
        }

        await this.prisma.role.update({
            where: {
                id: roleId,
            },
            data: {
                deletedAt: null,
            },
        });

        // Delete cache
        await this.cache.delete('role');

        return role;
    }
}
