import { HttpException, HttpStatus, Injectable } from '@nestjs/common';
import { Prisma } from '@prisma/client';
import * as argon from 'argon2';
import * as dayjs from 'dayjs';
import { MemberSharedService } from 'src/member/member-shared.service';
import memberData from 'src/data/member';
import { PrismaService } from 'src/prisma/prisma.service';
import { MemberQuery } from 'src/types/member';
import { conditionalReturn } from 'src/utils';

@Injectable()
export class MemberService {
    constructor(
        private prisma: PrismaService,
        private shared: MemberSharedService,
    ) {}

    async getMemberList(query: MemberQuery) {
        const { page, pageSize, sortField, sortOrder, fullName, email, phoneNumber, dateOfBirth, status, createdAt } = query;

        // Find member where params
        const memberWhereOptions: Prisma.MemberWhereInput = {
            deletedAt: null,
            ...conditionalReturn(!!fullName, {
                OR: [{ fullName: { mode: 'insensitive', contains: fullName } }, { preferredName: { mode: 'insensitive', contains: fullName } }],
            }),
            ...conditionalReturn(!!email, { email: { mode: 'insensitive', contains: email } }),
            ...conditionalReturn(!!phoneNumber, { phoneNumber: { mode: 'insensitive', contains: phoneNumber } }),
            ...conditionalReturn(!!dateOfBirth, { dateOfBirth: dayjs(dateOfBirth).format('YYYY-MM-DD') }),
            ...conditionalReturn(!!status, {
                status,
            }),
            ...conditionalReturn(createdAt && createdAt.length === 2, {
                createdAt: {
                    gte: createdAt && createdAt.length === 2 && new Date(dayjs(createdAt[0]).startOf('day').toISOString()),
                    lte: createdAt && createdAt.length === 2 && new Date(dayjs(createdAt[1]).endOf('day').toISOString()),
                },
            }),
        };

        const memberCount = await this.prisma.member.count({
            where: memberWhereOptions,
        });

        // If page is out of range, return the last page
        const currentPage = this.prisma.pageCounter(memberCount, page, pageSize);

        // Get filtered member and filtered member count
        const member = await this.prisma.member.findMany({
            take: pageSize,
            skip: (currentPage - 1) * pageSize,
            orderBy: {
                [!sortField ? 'createdAt' : sortField]: sortOrder ?? 'asc',
            },
            where: memberWhereOptions,
            select: {
                ...this.prisma.createSelect(memberData.exclude(['deletedAt'])),
                tokens: {
                    select: {
                        ...this.prisma.createSelect(['type', 'token', 'expiredAt', 'usedAt']),
                    },
                    orderBy: {
                        createdAt: 'desc',
                    },
                    take: 1,
                },
            },
        });

        // Map member password to true if password exists, false if not
        const mappedMember = member.map((item) => {
            const hasPassword = !!item.password;
            delete item.password;
            return {
                ...item,
                password: hasPassword,
            };
        });

        return {
            total: memberCount,
            rows: mappedMember,
            page: currentPage,
        };
    }

    async getExportMemberList() {
        const member = await this.prisma.member.findMany({
            where: {
                deletedAt: null,
            },
            select: {
                ...this.prisma.createSelect(['fullName', 'phoneNumber', 'dateOfBirth', 'address', 'country', 'state', 'city', 'createdAt']),
            },
        });

        return member;
    }

    async getMemberById(memberId: string) {
        const member = await this.prisma.member.findFirst({
            where: {
                id: memberId,
                deletedAt: null,
            },
            select: {
                ...this.prisma.createSelect(memberData.exclude(['deletedAt'])),
                tokens: {
                    select: {
                        ...this.prisma.createSelect(['type', 'token', 'expiredAt', 'usedAt']),
                    },
                    orderBy: {
                        createdAt: 'desc',
                    },
                    take: 1,
                },
            },
        });

        if (!member) {
            throw new HttpException('api-messages:member-not-found', HttpStatus.NOT_FOUND);
        }

        // Map member password to true if password exists, false if not
        const hasPassword = !!member.password;
        delete member.password;

        return {
            ...member,
            password: hasPassword,
        };
    }

    async getMemberBirthdayList() {
        const memberBirthday = await this.prisma.member.findMany({
            where: {
                status: 'ACTIVE',
                deletedAt: null,
            },
            select: {
                ...this.prisma.createSelect(['id', 'fullName', 'preferredName', 'dateOfBirth', 'address', 'phoneNumber']),
            },
        });

        return memberBirthday;
    }

    async updateMemberStatus(memberId: string) {
        const member = await this.prisma.member.findFirst({
            where: {
                id: memberId,
                deletedAt: null,
            },
            select: {
                ...this.prisma.createSelect(['id', 'status']),
            },
        });

        if (!member) {
            throw new HttpException('api-messages:member-not-found', HttpStatus.NOT_FOUND);
        }

        const updatedMember = await this.prisma.member.update({
            where: {
                id: member.id,
            },
            data: {
                status: member.status === 'ACTIVE' ? 'INACTIVE' : 'ACTIVE',
            },
            select: this.prisma.createSelect(memberData.exclude(['password', 'deletedAt'])),
        });

        return updatedMember;
    }

    async updateMemberProfile(memberId: string, body: UpdateMemberProfileDto) {
        const { fullName, preferredName, address, dateOfBirth, source, preferredLanguage, country, state, city } = body;

        const member = await this.prisma.member.findFirst({
            where: {
                id: memberId,
                deletedAt: null,
            },
            select: {
                ...this.prisma.createSelect(['id']),
            },
        });

        if (!member) {
            throw new HttpException('api-messages:member-not-found', HttpStatus.NOT_FOUND);
        }

        const updatedMember = await this.prisma.member.update({
            where: {
                id: member.id,
            },
            data: {
                fullName,
                preferredName,
                address,
                dateOfBirth,
                source,
                preferredLanguage,
                country: city === '' ? country : 'Malaysia',
                state,
                city,
            },
            select: this.prisma.createSelect(memberData.exclude(['password', 'deletedAt'])),
        });

        return updatedMember;
    }

    async updateMemberEmail(memberId: string, body: UpdateMemberEmailDto) {
        const member = await this.prisma.member.findFirst({
            where: {
                id: memberId,
                deletedAt: null,
            },
            select: {
                ...this.prisma.createSelect(['id']),
            },
        });

        if (!member) {
            throw new HttpException('api-messages:member-not-found', HttpStatus.NOT_FOUND);
        }

        // Check existing email
        const { isCreated } = await this.shared.emailCheck(body.email, memberId);

        // If email already exists, throw error
        if (isCreated) {
            throw new HttpException('api-messages:email-already-exists', HttpStatus.BAD_REQUEST);
        }

        const updatedMember = await this.prisma.member.update({
            where: {
                id: member.id,
            },
            data: {
                email: body.email,
            },
            select: this.prisma.createSelect(memberData.exclude(['password', 'deletedAt'])),
        });

        return updatedMember;
    }

    async updateMemberPhoneNumber(memberId: string, body: UpdateMemberPhoneNumberDto) {
        const member = await this.prisma.member.findFirst({
            where: {
                id: memberId,
                deletedAt: null,
            },
            select: {
                ...this.prisma.createSelect(['id']),
            },
        });

        if (!member) {
            throw new HttpException('api-messages:member-not-found', HttpStatus.NOT_FOUND);
        }

        const updatedMember = await this.prisma.member.update({
            where: {
                id: member.id,
            },
            data: {
                phoneNumber: body.phoneNumber,
            },
            select: this.prisma.createSelect(memberData.exclude(['password', 'deletedAt'])),
        });

        return updatedMember;
    }

    async updateMemberPassword(memberId: string, body: UpdateMemberPasswordDto) {
        const member = await this.prisma.member.findFirst({
            where: {
                id: memberId,
                deletedAt: null,
            },
            select: {
                ...this.prisma.createSelect(['id']),
            },
        });

        if (!member) {
            throw new HttpException('api-messages:member-not-found', HttpStatus.NOT_FOUND);
        }

        // Hash password
        const hashedPassword = await argon.hash(body.password);

        const updatedMember = await this.prisma.member.update({
            where: {
                id: member.id,
            },
            data: {
                password: hashedPassword,
            },
            select: this.prisma.createSelect(memberData.exclude(['password', 'deletedAt'])),
        });

        return updatedMember;
    }

    async deleteMember(memberId: string) {
        const member = await this.prisma.member.findFirst({
            where: {
                id: memberId,
                deletedAt: null,
            },
            select: {
                ...this.prisma.createSelect(['id']),
            },
        });

        if (!member) {
            throw new HttpException('api-messages:member-not-found', HttpStatus.NOT_FOUND);
        }

        const deletedMember = await this.prisma.member.update({
            where: {
                id: member.id,
            },
            data: {
                deletedAt: dayjs().toDate(),
            },
            select: this.prisma.createSelect(memberData.exclude(['password', 'deletedAt'])),
        });

        return deletedMember;
    }
}
