import { HttpException, HttpStatus, Injectable } from '@nestjs/common';
import { AccountStatus, BookTokenSubscriptionStatus, Prisma } from '@prisma/client';
import * as dayjs from 'dayjs';
import { PrismaService } from 'src/prisma/prisma.service';
import { SiteSettingService } from 'src/staff/site-setting/site-setting.service';
import { Pagination } from 'src/types';
import { MediaService } from 'src/media/media.service';
import { RequestBookTokenDto, UploadProofOfPaymentDto } from './book-token.dto';

@Injectable()
export class MemberBookTokenService {
    constructor(
        private readonly prisma: PrismaService,
        private readonly siteSettingService: SiteSettingService,
        private readonly media: MediaService,
    ) {}

    async getBookTokenSubscriptionList(memberId: string, query: Pagination) {
        const { page, pageSize, sortField, sortOrder } = query;

        const whereOptions: Prisma.BookTokenSubscriptionWhereInput = {
            memberId,
            deletedAt: null,
        };

        const subscriptionCount = await this.prisma.bookTokenSubscription.count({
            where: whereOptions,
        });

        const currentPage = this.prisma.pageCounter(subscriptionCount, page, pageSize);

        const bookTokenSubscriptionList = await this.prisma.bookTokenSubscription.findMany({
            where: whereOptions,
            skip: (currentPage - 1) * pageSize,
            take: pageSize,
            orderBy: {
                [!sortField ? 'createdAt' : sortField]: sortOrder ?? 'asc',
            },
            select: {
                ...this.prisma.createSelect([
                    'id',
                    'price',
                    'displayId',
                    'paymentStatus',
                    'reason',
                    'quantity',
                    'numberOfBookTokens',
                    'createdAt',
                    'paymentDate',
                ]),
                bookTokenSubscriptionMedias: {
                    select: this.prisma.createSelect(['mediaId', 'media']),
                },
                package: {
                    select: this.prisma.createSelect(['id', 'name', 'price', 'numberOfBookToken', 'description']),
                },
            },
        });

        return {
            total: subscriptionCount,
            rows: bookTokenSubscriptionList,
            page: currentPage,
        };
    }

    async getBookTokenSubscriptionById(bookTokenId: string) {
        const bookTokenSubscription = await this.prisma.bookTokenSubscription.findUnique({
            where: {
                id: bookTokenId,
            },
            select: {
                ...this.prisma.createSelect(['id', 'price', 'displayId', 'paymentStatus', 'reason', 'quantity', 'expiredAt']),
                bookTokenSubscriptionMedias: {
                    select: {
                        media: {
                            select: this.prisma.createSelect(['id', 'name', 'type', 'key']),
                        },
                    },
                },
                member: {
                    select: this.prisma.createSelect(['id', 'fullName', 'preferredName']),
                },
                package: {
                    select: this.prisma.createSelect(['name', 'description']),
                },
            },
        });

        if (!bookTokenSubscription) {
            throw new HttpException('api-messages:book-token-not-found', HttpStatus.NOT_FOUND);
        }

        return bookTokenSubscription;
    }

    async requestBookToken(memberId: string, body: RequestBookTokenDto) {
        const { selectedPackage, quantity, totalAmount } = body;

        const isMemberExist = await this.prisma.member.findFirst({
            where: {
                id: memberId,
                deletedAt: null,
                status: AccountStatus.ACTIVE,
            },
            select: {
                id: true,
            },
        });

        if (!isMemberExist) {
            throw new HttpException('api-messages:member-not-found', HttpStatus.NOT_FOUND);
        }

        const selectedPackageData = await this.siteSettingService.getSinglePackage(selectedPackage);

        /* Create Subscription for Book Token */
        const bookTokenSubscription = await this.prisma.bookTokenSubscription.create({
            data: {
                memberId,
                price: totalAmount,
                numberOfBookTokens: selectedPackageData.numberOfBookToken * quantity,
                quantity,
                packageId: selectedPackageData.id,
            },
            select: { id: true },
        });

        await this.updateDisplayId(bookTokenSubscription.id);

        return bookTokenSubscription;
    }

    async cancelBookTokenRequest(memberId: string, bookTokenSubscriptionId: string) {
        const member = await this.prisma.member.findUnique({
            where: {
                id: memberId,
            },
            select: {
                id: true,
            },
        });

        if (!member) {
            throw new HttpException('api-messages:member-not-found', HttpStatus.NOT_FOUND);
        }

        const bookToken = await this.prisma.bookTokenSubscription.findUnique({
            where: {
                id: bookTokenSubscriptionId,
                deletedAt: null,
            },
        });

        if (!bookToken) {
            throw new HttpException('api-messages:book-token-not-found', HttpStatus.NOT_FOUND);
        }

        const response = await this.prisma.bookTokenSubscription.update({
            where: {
                id: bookTokenSubscriptionId,
            },
            data: {
                deletedAt: dayjs().toDate(),
            },
            select: {
                id: true,
            },
        });

        return response;
    }

    /* Upload Media for Proof Of Payment */
    async uploadPaymentProofMedia(file: Express.Multer.File) {
        const fileResponse = await this.media.upload(file, { isPublic: false });

        const mediaResponse = await this.prisma.media.create({
            data: fileResponse,
            select: this.prisma.createSelect(['id', 'name', 'type', 'key']),
        });

        return mediaResponse;
    }

    /* Upload Proof Of Payment */
    async uploadProofOfPayment(memberId: string, bookTokenSubscriptionId: string, body: UploadProofOfPaymentDto) {
        const member = await this.prisma.member.findUnique({
            where: {
                id: memberId,
            },
            select: {
                id: true,
            },
        });

        if (!member) {
            throw new HttpException('api-messages:member-not-found', HttpStatus.NOT_FOUND);
        }

        const bookToken = await this.prisma.bookTokenSubscription.findUnique({
            where: {
                id: bookTokenSubscriptionId,
                deletedAt: null,
            },
        });

        if (!bookToken) {
            throw new HttpException('api-messages:book-token-not-found', HttpStatus.NOT_FOUND);
        }

        const response = await this.prisma.bookTokenSubscription.update({
            where: {
                id: bookTokenSubscriptionId,
            },
            data: {
                paymentDate: body.paymentDate,
                paymentStatus: BookTokenSubscriptionStatus.PENDING,
                bookTokenSubscriptionMedias: {
                    deleteMany: {},
                    createMany: {
                        data: body.medias.map((media) => ({
                            mediaId: media.mediaId,
                        })),
                    },
                },
            },
            select: {
                id: true,
            },
        });

        return response;
    }

    /* Update Display Id */
    private async updateDisplayId(bookTokenSubscriptionId: string) {
        const response = await this.prisma.bookTokenSubscription.findFirst({
            where: {
                id: bookTokenSubscriptionId,
            },
            select: {
                refNo: true,
            },
        });

        const ref = response.refNo.toString().padStart(2, '0');
        const displayId = `BR-${dayjs().format('YYMMDD')}-${ref}`;

        const updateResponse = await this.prisma.bookTokenSubscription.update({
            where: {
                id: bookTokenSubscriptionId,
            },
            data: {
                displayId,
            },
            select: {
                id: true,
            },
        });

        return !!updateResponse;
    }
}
